# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import pymongo
from django.conf import settings

from common.db.mongo import databases
from travel.rasp.info_center.info_center.suburban_notify.db import get_point_by_key
from travel.rasp.info_center.info_center.suburban_notify.utils import InitableSlots, get_interval

log = logging.getLogger(__name__)


class Importance(object):
    ANY = 'any'
    ONLY_IMPORTANT = 'only_important'


class Frequency(object):
    EVERY_DAY = 'every_day'
    FIRST_DAY = 'first_day'


class Subscription(InitableSlots):
    __slots__ = [
        'uid',
        'point_from_key',
        'point_to_key',
        'interval_from',
        'interval_to',
        'importance',
        'frequency',
    ]

    slots_defaults = {
        'importance': Importance.ANY,
        'frequency': Frequency.FIRST_DAY,
    }

    indexes_inited = False

    @classmethod
    def init_indexes(cls):
        if not cls.indexes_inited:
            cls.get_collection().create_index(
                [
                    ('uid', pymongo.ASCENDING),
                ],
            )
            cls.get_collection().create_index(
                [
                    ('point_to_key', pymongo.ASCENDING),
                    ('point_from_key', pymongo.ASCENDING),
                    ('uid', pymongo.ASCENDING),
                ],
            )
            cls.indexes_inited = True

    def __init__(self, *args, **kwargs):
        super(Subscription, self).__init__(*args, **kwargs)
        self.init_indexes()

    def __str__(self):
        return '{} -> {}'.format(self.point_from_key, self.point_to_key)

    def __repr__(self):
        return self.__str__()

    @classmethod
    def get_collection(cls):

        return databases[settings.SUBURBAN_NOTIFICATION_DATABASE_NAME].suburban_subscriptions

    def get_interval(self, day):
        point_from = get_point_by_key(self.point_from_key)
        return get_interval(day, self.interval_from, self.interval_to, point_from.pytz)

    @property
    def key(self):
        return {
            'uid': self.uid,
            'point_from_key': self.point_from_key,
            'point_to_key': self.point_to_key
        }

    def save(self):
        self.get_collection().update(
            self.key,
            {
                '$set': {
                    'importance': self.importance,
                    'frequency': self.frequency,
                    'interval_from': self.interval_from,
                    'interval_to': self.interval_to
                }
            },
            upsert=True
        )

    def delete(self):
        self.get_collection().delete_one(self.key)

    @classmethod
    def subscription_from_dict(cls, **kwargs):
        kwargs.pop('_id')
        return Subscription(**kwargs)

    @classmethod
    def get_subscription(cls, uid, point_from_key, point_to_key):
        subscription = cls.get_collection().find_one({
            'uid': uid,
            'point_from_key': point_from_key,
            'point_to_key': point_to_key
        })

        if subscription:
            return cls.subscription_from_dict(**subscription)

    @classmethod
    def get_subscriptions(cls, limit=0, **kwargs):
        subscriptions = []
        for subscription in cls.get_collection().find(kwargs).limit(limit):
            subscriptions.append(cls.subscription_from_dict(**subscription))

        return subscriptions

    @classmethod
    def get_subscriptions_stat(cls):
        coll = cls.get_collection()
        return {
            'total': coll.count(),
            'users': len(list(coll.distinct('uid'))),
        }
