# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from contextlib import contextmanager
from datetime import datetime, time, timedelta
from functools import partial

from travel.rasp.library.python.common23.logging import log_run_time


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


@contextmanager
def disable_logs(level=logging.INFO):
    import logging
    root_logger = logging.getLogger()
    prev_level = root_logger.level
    try:
        root_logger.level = level
        yield
    finally:
        root_logger.level = prev_level


def run_precache():
    with disable_logs():
        from common.precache.backend import setup_precache
        with log_run_time('setup precache'):
            setup_precache(log)


def get_chunks_indexes(objs, chunk_size):
    indexes_pairs = []

    i = 0
    while i < len(objs):
        next_i = i + chunk_size
        indexes_pairs.append([i, next_i])
        i = next_i

    indexes_pairs[-1][1] = len(objs)
    return indexes_pairs


def cached(func):
    cache = {}

    def wrapper(*args, **kwargs):
        key = (tuple(args), tuple((k, v) for k, v in kwargs.items()))

        if key not in cache:
            result = func(*args, **kwargs)
            cache[key] = result

        return cache[key]

    wrapper.cache = cache

    return wrapper


def get_interval(day, minutes_from, minutes_to, tz):
    dt = tz.localize(datetime.combine(day, time()))
    interval_from = dt + timedelta(minutes=minutes_from)
    interval_to = dt + timedelta(minutes=minutes_to)

    return interval_from, interval_to


class InitableSlots(object):
    __slots__ = ()
    slots_defaults = {}

    def __init__(self, *args, **kwargs):
        filled_slots = []
        for slot_ind, value in enumerate(args):
            slot_name = self.__slots__[slot_ind]
            setattr(self, slot_name, value)

            if slot_name in kwargs:
                raise ValueError('Duplicate arg: {}'.format(slot_name))

            filled_slots.append(slot_name)

        for slot_name, value in kwargs.items():
            setattr(self, slot_name, value)
            filled_slots.append(slot_name)

        for slot_name, value in self.slots_defaults.items():
            if slot_name not in filled_slots:
                setattr(self, slot_name, value)
