# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import re
from collections import defaultdict
from datetime import datetime

from django.contrib.auth.decorators import login_required
from django.http import JsonResponse
from django.shortcuts import redirect
from django.template.response import TemplateResponse, HttpResponse
from django.utils.decorators import method_decorator
from django.utils.encoding import force_text
from django.views.decorators.http import require_http_methods
from django.views.generic import View

from common.apps.info_center.models import Info, Message, ObjLinkType, Push
from common.apps.info_center.push import PushSender
from common.apps.info_center.yadm_news_item import YadmNewsItemSaver
from common.data_api.sup.client import SupClient
from travel.rasp.library.python.common23.date import environment

from .forms import InfoForm, PushForm, YadmNewsItemForm


MAX_LINKED_OBJECTS_GROUPS = 1


def get_field_values(post_data, field_regexp):
    for field, value in post_data.items():
        if not value:
            continue

        match = re.match(field_regexp, field)
        if match:
            yield list(match.groups()) + [value]


def parse_linked_objects(post_data, container_suffix=''):
    objs = defaultdict(list)
    iter_links = get_field_values(post_data, r'^idLinkedObjectsContainer{}(\d+)_(\w+)_(\d+)$'.format(container_suffix))
    for list_id_str, obj_type, obj_index, obj_id in iter_links:
        obj_link = Info.get_link_by_obj_id(obj_type, obj_id)
        objs[int(list_id_str) - 1].append(obj_link)

    linked_objects = [[] for _ in range(max(objs.keys()) + 1)] if objs else []
    for list_id, linked_objs in objs.items():
        linked_objects[list_id] = linked_objs

    return linked_objects


def parse_messages(post_data):
    msgs = defaultdict(dict)
    for msg_ind, text in get_field_values(post_data, r'^idMessagesContainer_(\d+)_text$'):
        msgs[msg_ind]['text'] = text

    for msg_ind, dt_created in get_field_values(post_data, r'^idMessagesContainer_(\d+)_dtCreated$'):
        msgs[msg_ind]['dt_created'] = datetime.strptime(dt_created, '%Y-%m-%d %H:%M')

    result = []
    for msg_id, msg in sorted(msgs.items(), key=lambda kv: kv[0]):
        msg.setdefault('dt_created', environment.now())
        result.append(Message(**msg))

    return result


def linked_objects_for_template(links):
    result = []
    for link in links:
        obj = Info.get_object_by_link(link)
        if obj:
            title = Info.get_linked_obj_title(link.obj_type, obj)
            result.append((link.obj_type, obj.id, title))
        else:
            result.append((link.obj_type, '', 'unknown obj with key "{}"'.format(link.obj_key)))

    return sorted(result, key=lambda o: (o[0], o[2]))


def messages_for_template(messages):
    return [
        {
            'type': 'message',
            'text': msg.text,
            'dtCreated': msg.dt_created.strftime('%Y-%m-%d %H:%M')
        }
        for msg in sorted(messages, key=lambda msg: msg.dt_created)
    ]


def pushes_for_template(pushes):
    result = []

    for push in sorted(pushes, key=lambda push: push.dt_created):
        push_data = {
            'type': 'push',
            'id': push.id,
            'title': push.title,
            'text': push.text,
            'url': push.url,
            'imageUrl': push.image_url,
            'policy': force_text(dict(Push.POLICIES)[push.policy]),
            'dtCreated': push.dt_created.strftime('%Y-%m-%d %H:%M'),
            'error': push.error,
        }
        push_data['json'] = json.dumps(push_data)

        result.append(push_data)

    return result


def yadm_news_for_template(yadm_news):
    result = []

    for news_item in sorted(yadm_news, key=lambda news_item: news_item.dt_created):
        yadm_news_item_data = {
            'type': 'yadm_news_item',
            'id': news_item.id,
            'title': news_item.title,
            'text': news_item.text,
            'importance': news_item.importance,
            'dtCreated': news_item.dt_created.strftime('%Y-%m-%d %H:%M'),
        }
        yadm_news_item_data['json'] = json.dumps(yadm_news_item_data)

        result.append(yadm_news_item_data)

    return result


def get_linked_obj_groups(linked_objects):
    # берем имеющиеся группы + добиваем до MAX_LINKED_OBJECTS_GROUPS групп, если не хватает
    return (
        [linked_objects_for_template(l) for l in linked_objects] +
        [[] for _ in range(MAX_LINKED_OBJECTS_GROUPS - len(linked_objects))]
    )


def get_obj_view_response(request, info_id=None, form=None, push_form=None, yadm_news_item_form=None, info=None):
    if not form:
        form = InfoForm(initial={'uuid': Info.generate_uuid()})

    if not push_form:
        push_form = PushForm(prefix='push')

    if not yadm_news_item_form:
        yadm_news_item_form = YadmNewsItemForm(prefix='yadm_news_item')

    feed = []
    if info:
        feed += messages_for_template(info.messages)
        feed += pushes_for_template(info.pushes)
        feed += yadm_news_for_template(info.yadm_news)

    feed = sorted(feed, key=lambda o: o['dtCreated'])

    return TemplateResponse(
        request,
        'info_center.html',
        {
            'form': form,
            'form_push': push_form,
            'form_yadm_news_item': yadm_news_item_form,
            'info_id': info_id,
            'linked_objs_groups': get_linked_obj_groups(info.linked_objects if info else []),
            'linked_objs_types': Info.get_linked_obj_types(),
            'default_linked_obj_type': ObjLinkType.DIRECTION,
            'feed': feed,
        }
    )


class InfoView(View):
    @method_decorator(login_required)
    def get(self, request, info_id):
        info = Info.objects.get(id=info_id)

        return get_obj_view_response(
            request, info_id,
            form=InfoForm(initial=info.to_mongo()),
            info=info
        )

    @method_decorator(login_required)
    def post(self, request, info_id):
        return info_change_or_create(request, info_id)


@login_required
def info_change_or_create(request, info_id):
    info = Info.objects.get(id=info_id) if info_id else None
    form = InfoForm(request.POST)

    if form.is_valid():
        if info:
            info.update(**form.cleaned_data)
        else:
            info = Info.objects.create(**form.cleaned_data)

        info.linked_objects = parse_linked_objects(request.POST)
        info.messages = parse_messages(request.POST)
        info.save()

        if request.POST.get('save_and_continue_button'):
            return redirect('info_obj', info.id)
        else:
            return redirect('info_list')
    else:
        return get_obj_view_response(request, info_id, form, info=info)


@login_required
@require_http_methods(['GET', 'POST'])
def info_add(request):
    if request.method == 'GET':
        return get_obj_view_response(request)

    return info_change_or_create(request, None)


@login_required
@require_http_methods(['POST'])
def info_remove(request, info_id):
    Info.objects.filter(id=info_id).delete()
    return redirect('info_list')


@login_required
@require_http_methods(['GET'])
def info_raw(request, info_id):
    info = Info.objects.get(id=info_id)
    return JsonResponse(json.loads(info.to_json()))


@login_required
@require_http_methods(['GET'])
def info_list(request):
    infos = sorted(Info.objects.all(), key=lambda o: (o.services, o.id), reverse=True)

    return TemplateResponse(request, 'info_center_list.html', {'infos': infos})


@login_required
@require_http_methods(['POST'])
def send_push(request, info_id):
    push_form = PushForm(request.POST, prefix='push')
    if push_form.is_valid():
        try:
            sender = PushSender(info_id, user=request.user.username)
            sender.send(**push_form.cleaned_data)
        except SupClient.InvalidRequest as e:
            msg = '<br>'.join([e.msg] + e.errors)
            msg += '<br><a href="https://push-beta.n.yandex-team.ru/docs/api-guide.html#resources-push-create">Дока</a>'
            return HttpResponse(msg, status=500)
        except Exception as e:
            return HttpResponse(str(e), status=500)

        return redirect('info_obj', info_id)
    else:
        info = Info.objects.get(id=info_id)
        return get_obj_view_response(
            request, info_id,
            form=InfoForm(initial=info.to_mongo()),
            push_form=push_form,
            info=info
        )


@login_required
@require_http_methods(['POST'])
def add_yadm_news_item(request, info_id):
    yadm_news_item_form = YadmNewsItemForm(request.POST, prefix='yadm_news_item')
    if yadm_news_item_form.is_valid():
        try:
            saver = YadmNewsItemSaver(info_id)
            saver.save(**yadm_news_item_form.cleaned_data)
        except Exception as e:
            return HttpResponse(str(e), status=500)

        return redirect('info_obj', info_id)
    else:
        info = Info.objects.get(id=info_id)
        return get_obj_view_response(
            request, info_id,
            form=InfoForm(initial=info.to_mongo()),
            yadm_news_item_form=yadm_news_item_form,
            info=info
        )


@login_required
def gen_info_uid(request):
    return JsonResponse({'value': str(Info.generate_uuid())})
