# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import os
import socket
import traceback

import pymongo
from django.db import connection
from django.http import HttpResponse, JsonResponse
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from common.db.mongo import database
from common.apps.info_center.getter import get_info


@require_http_methods(['GET'])
def get_info_view(request):
    service = request.GET['service']
    page = request.GET['page']
    national_version = request.GET.get('national_version', 'ru')

    infos = get_info([service], page, [national_version])

    response = []
    for info in infos:
        response.append(json.loads(info.to_json()))

    return JsonResponse({'infos': response})


def safe(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception:
            return traceback.format_exc()

    return wrapper


@safe
def get_hostname():
    return socket.gethostname()


@safe
def get_mysql_host():
    c = connection.cursor()
    c.execute('select @@hostname;')
    c.execute("show variables like 'hostname';")
    return list(c.fetchall())[0][1]


@safe
def get_mongo_host():
    try:
        return [database.client.address]
    except pymongo.errors.InvalidOperation:
        return list(database.client.nodes)


@safe
def get_qloud_env():
    return {k: v for k, v in os.environ.items() if 'qloud' in k.lower()}


@safe
def get_user(request):
    return str(request.user)


@require_http_methods(['GET'])
def ping(request):
    return JsonResponse({
        'mysql_host': get_mysql_host(),
        'mongo_hosts': get_mongo_host(),
    })


@require_http_methods(['GET'])
def env(request):
    return JsonResponse({
        'host': get_hostname(),
        'pid': os.getpid(),
        'qloud_env': get_qloud_env(),
        'user': get_user(request),
    })


@csrf_exempt
@require_http_methods(['POST'])
def deploy_hook(request):
    body = request.POST
    print(body)

    return HttpResponse('ok')


@require_http_methods(['GET'])
def version(request):
    return HttpResponse('version1: {}, {}'.format(socket.gethostname(), dict(request.GET)))
