# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
from datetime import datetime

from django.http import JsonResponse, HttpResponse
from django.views.decorators.http import require_http_methods
from marshmallow import Schema, fields, validate

from common.settings.utils import bool_converter
from common.utils.blackbox_wrapper import get_blackbox_oauth_args, get_blackbox_oauth_info, SessionInvalid
from travel.rasp.info_center.info_center.suburban_notify.main import get_subscription_changes
from travel.rasp.info_center.info_center.suburban_notify.subscriptions.models import Subscription, Frequency, Importance


log = logging.getLogger(__name__)

RESPONSE_200 = JsonResponse({'status': 'done'})


class SubscriptionKeySchema(Schema):
    point_from_key = fields.String(required=True)
    point_to_key = fields.String(required=True)


class SubscriptionSchema(SubscriptionKeySchema):
    interval_from = fields.Int(required=True)
    interval_to = fields.Int(required=True)
    importance = fields.String(required=True, validate=validate.OneOf([Importance.ANY, Importance.ONLY_IMPORTANT]))
    frequency = fields.String(required=True, validate=validate.OneOf([Frequency.FIRST_DAY, Frequency.EVERY_DAY]))


def get_blackbox_info(request, **extra_bb_args):
    bb_args = get_blackbox_oauth_args(request)
    bb_args.update(extra_bb_args)

    return get_blackbox_oauth_info(**bb_args)


def get_query_bb(request, schema=None):
    query, error_response = {}, None

    try:
        if schema:
            query, errors = schema().load(request.GET.dict())
            if errors:
                raise Exception()
        blackbox_info = get_blackbox_info(request)
        query['uid'] = blackbox_info.uid
    except SessionInvalid:
        error_response = JsonResponse({'error': 'Unauthorized'}, status=403)
    except Exception as ex:
        log.error('failed {}'.format(repr(ex)))
        error_response = JsonResponse({'error': 'bad request'}, status=400)

    return query, error_response


def get_subscription_dict(subscription):
    return {
        attr: getattr(subscription, attr) for attr in
        ['point_from_key', 'point_to_key', 'interval_from', 'interval_to', 'importance', 'frequency']
    } if subscription else {}


@require_http_methods(['GET'])
def modify_subscription(request):
    query, error_response = get_query_bb(request, SubscriptionSchema)
    if error_response:
        return error_response

    subscription = Subscription(**query)
    subscription.save()

    return RESPONSE_200


@require_http_methods(['GET'])
def delete_subscription(request):
    query, error_response = get_query_bb(request, SubscriptionKeySchema)
    if error_response:
        return error_response

    subscription = Subscription(**query)
    subscription.delete()

    return RESPONSE_200


@require_http_methods(['GET'])
def get_subscription(request):
    query, error_response = get_query_bb(request, SubscriptionKeySchema)
    if error_response:
        return error_response

    subscription = Subscription.get_subscription(**query)
    subscription_data = get_subscription_dict(subscription)

    return JsonResponse(subscription_data)


@require_http_methods(['GET'])
def get_all_subscriptions(request):
    # query, error_response = get_query_bb(request)
    # if error_response:
    #     return error_response
    #
    # subscriptions = Subscription.get_subscriptions(**query)
    # subscriptions_data = [get_subscription_dict(subscription) for subscription in subscriptions]

    # Отключаем отображение подписок, скоро выпилим совсем
    # https://st.yandex-team.ru/SUBURBAN-3214
    subscriptions_data = []

    return JsonResponse(subscriptions_data, safe=False)


@require_http_methods(['GET'])
def get_changes(request):
    # query, error_response = get_query_bb(request, SubscriptionKeySchema)
    # if error_response:
    #     return error_response
    #
    # subs_changes = ChangesFinderWithStorage().get_subs_changes(**query)
    # subscription = Subscription.get_subscription(**query)

    # Отключаем отображение подписок, скоро выпилим совсем
    # https://st.yandex-team.ru/SUBURBAN-3214
    changes_data = {}

    # if subscription:
    #     changes_data['subscription'] = {
    #         'point_from_key': subscription.point_from_key,
    #         'point_to_key': subscription.point_to_key,
    #         'interval_to': subscription.interval_to,
    #         'interval_from': subscription.interval_from,
    #         'importance': subscription.importance,
    #         'frequency': subscription.frequency
    #     }
    #
    # if subs_changes:
    #     changes_data['changes'] = []
    #     text_generator = TextGenerator()
    #     text_generator.load_objects(subs_changes)
    #     for sub_changes in subs_changes:
    #         sub_changes_dict = {'changes_list': [], 'date': str(sub_changes.calc_date.date())}
    #         for change in sorted(set(sub_changes.changes), key=lambda ch: ch.sort_time()):
    #             text = text_generator.get_text_for_change(change)
    #             sub_changes_dict['changes_list'].append({
    #                 'title': text['thread'],
    #                 'description': text['description'],
    #                 'is_first_run_day': change.is_first_run_day(),
    #                 'is_important': change.is_important()
    #             })
    #         changes_data['changes'].append(sub_changes_dict)

    return HttpResponse(json.dumps(changes_data, ensure_ascii=False), content_type='application/json')


@require_http_methods(['GET'])
def get_subsciption_changes(request):
    """
    Generate changes for subscription + some debug info

    Dev method.
    http://mondev.sas.yp-c.yandex.net:8005/subscription/dev/get_changes?point_from=s9600212&point_to=s2006004&minutes_from=0&minutes_to=1440&day=2019-03-11&frequency=first_day&importance=only_important
    """
    point_from = request.GET['point_from']
    point_to = request.GET['point_to']
    minutes_from = int(request.GET['minutes_from'])
    minutes_to = int(request.GET['minutes_to'])
    day_str = request.GET['day']
    day = datetime.strptime(day_str, '%Y-%m-%d')
    importance = request.GET.get('importance')
    frequency = request.GET.get('frequency')
    only_new = bool_converter(request.GET.get('only_new', False))

    sub_changes = get_subscription_changes(
        point_from, point_to, minutes_from, minutes_to, day,
        importance=importance, frequency=frequency,
        only_new=only_new,
    )

    return JsonResponse({
        'push': sub_changes['push'],
        'texts': sub_changes['texts'],
        'changes': sub_changes['changes'],
    })
