# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.views.decorators.http import require_http_methods
from django.http import JsonResponse

from common.apps.info_center.models import Info, ObjLinkType
from common.models.geo import Settlement, Station, ExternalDirection
from common.xgettext.common import xgettext_month_genitive


log = logging.getLogger(__name__)


@require_http_methods(['GET'])
def yadm_news(request):
    """
    Список новостей для ЯдМ (новостная лента Яндекс)
    https://st.yandex-team.ru/RASPFRONT-8124
    Возвращает все новости с id > last_id
    """
    last_id = int(request.GET.get('last_id', '0'))
    infos = Info.objects.filter(yadm_news__id__gt=last_id)

    yadm_pairs = []
    for info in infos:
        for new_item in info.yadm_news:
            if new_item.id > last_id:
                teaser_settlements = _get_teaser_settlements(info)
                yadm_pairs.append((new_item, teaser_settlements))

    yadm_pairs.sort(key=lambda pair: pair[0].id)

    return JsonResponse({
        'news': [_yadm_news_item_for_json(new_item, settlements) for new_item, settlements in yadm_pairs]
    })


def _get_teaser_settlements(info):
    """
    Получение городов по всем привязанным объектам (города, станции, направления)
    """
    if not info.linked_objects:
        return set()

    settlements = set()
    for ob in info.linked_objects[0]:
        if ob.obj_type == ObjLinkType.SETTLEMENT:
            try:
                settlements.add(Settlement.objects.get(id=ob.obj_key))
            except Settlement.DoesNotExist:
                log.error('Settlement with id={} not found'.format(ob.obj_key))

        elif ob.obj_type == ObjLinkType.STATION:
            try:
                station = Station.objects.get(id=ob.obj_key)
                if station.settlement:
                    settlements.add(station.settlement)
            except Station.DoesNotExist:
                log.error('Station with id={} not found'.format(ob.obj_key))

        elif ob.obj_type == ObjLinkType.DIRECTION:
            try:
                ex_direction = ExternalDirection.objects.get(id=ob.obj_key)
                if ex_direction.base_station.settlement:
                    settlements.add(ex_direction.base_station.settlement)
            except ExternalDirection.DoesNotExist:
                log.error('External direction with id={} not found'.format(ob.obj_key))

    return settlements


def _get_geo_ids(settlements):
    return [s._geo_id for s in settlements if s._geo_id]


def _get_main_settlement(settlements):
    """
    Получение наиболее значимого города из списка (с самым маленьким majority.id)
    """
    main_settlement = None
    for s in settlements:
        if not main_settlement or main_settlement.majority_id > s.majority_id:
            main_settlement = s
    return main_settlement


def _get_full_text(news_item, settlements):
    """
    Формирование текста новости
    Формируем только на русском языке
    """
    dt_string = '{} {}, {}'.format(
        news_item.dt_created.strftime('%-d'),
        xgettext_month_genitive(news_item.dt_created.month, 'ru').lower(),
        news_item.dt_created.strftime('%-H:%M')
    )

    main_settlement = _get_main_settlement(settlements)
    settlement_title = '{}. '.format(main_settlement.L_title()) if main_settlement else ''

    return '{}{}. ЯНДЕКС.РАСПИСАНИЯ. {}'.format(
        settlement_title,
        dt_string,
        news_item.text
    )


def _yadm_news_item_for_json(news_item, settlements):
    return {
        'id': news_item.id,
        'geoIds': _get_geo_ids(settlements),
        'importance': news_item.importance,
        'title': news_item.title,
        'text': _get_full_text(news_item, settlements),
        'code': 'suburban_message'
    }
