# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os

from rasp_vault.api import get_secret

from common.settings import *  # noqa
from common.settings.configuration import BaseConfigurator, Configuration
from common.settings.utils import configure_raven, get_app_version
from travel.rasp.library.python.common23.logging.configure import configure_default_logging


STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder'
]

TEMPLATES = [
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': False,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.contrib.auth.context_processors.auth',
                'django.template.context_processors.debug',
                'django.template.context_processors.media',
                'django.template.context_processors.request',
                'django.template.context_processors.static',
                'django.contrib.messages.context_processors.messages',
            ]
        },
    },
]

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'

PROJECT_PATH = os.path.abspath(os.path.join(os.path.dirname(__file__), ''))

LOG_PATH = '/var/log/app'

DEBUG = False

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

ALLOWED_HOSTS = ['localhost', '.yandex.ru', '.yandex.net', '.yandex-team.ru', '.yandex-team.net']

ROOT_URLCONF = 'travel.rasp.info_center.info_center.urls'

ADMINS = []


INSTALLED_APPS = COMMON_INSTALLED_APPS + COMMON_CONTRIB_APPS + [  # noqa
    'django_opentracing',
    'django.contrib.staticfiles',
    'django.contrib.admin',
    'django.contrib.humanize',
    'django.contrib.sessions',
    'django.contrib.sites',
    'raven.contrib.django.raven_compat',
    'travel.rasp.info_center.info_center',
]

MIDDLEWARE_CLASSES = [
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.request_activator.RequestActivator',
    'common.middleware.ping.PingResponder',
    'common.middleware.now.Now',

    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.middleware.gzip.GZipMiddleware',
    'django.middleware.security.SecurityMiddleware',

    'common.middleware.yateamuser.YaTeamUserAuth',
]

STATIC_ROOT = '/app/static/'

DOMAIN_TO_NATIONAL_VERSION = {
    'ru': 'ru',
    'ua': 'ua',
    'tr': 'tr'
}

# nginx (в т.ч. балансеры в qloud) ходит в приложение по http,
# при этом ставит HTTP_X_FORWARDED_PROTO, если изначальное соединение было по https.
# Эта настройка показывает Джанге, какой хедер читать, чтобы узнать, что соединение секьюрное.
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')


SUBURBAN_NOTIFICATION_DATABASE_NAME = 'suburban_notification'

PKG_VERSION = get_app_version()


class InfoCenterConfigurator(BaseConfigurator):
    def apply_base(self, settings):
        settings['TVM_DESTINATIONS'] = ['TVM_BLACKBOX', 'TVM_BLACKBOX_YATEAM']

        settings['MONGO_DATABASES'][SUBURBAN_NOTIFICATION_DATABASE_NAME] = {
            'port': 27018,
            'db': SUBURBAN_NOTIFICATION_DATABASE_NAME,
            'user': 'suburban',
            'options': {'socketTimeoutMS': 60000}
        }

        base_log_level = os.environ.get('RASP_BASE_LOG_LEVEL', 'INFO')
        format_with_context = '%(levelname)s %(asctime)s %(context)s %(process)d %(name)s %(message)s'
        settings['LOGGING'] = configure_default_logging(settings, base_log_level, format_with_context,
                                                        jaeger_tracing=True)

    def apply_testing(self, settings):
        settings['TVM_SERVICE_ID'] = 2002606

        notifications_db = settings['MONGO_DATABASES'][SUBURBAN_NOTIFICATION_DATABASE_NAME]
        notifications_db['password'] = get_secret('rasp-infocenter-testing.SUBURBAN_NOTIFICATION_MONGO_PASSWORD')
        notifications_db['dbaas_id'] = 'mdbj7l7cfjedppr3p6t8'
        notifications_db['fallback_hosts'] = [
            'vla-ogdt7yvr1gpae2pa.db.yandex.net'
        ]

        settings['BLACKBOX_URL'] = 'http://pass-test.yandex.ru/blackbox'

        configure_raven(settings, 'info_center')

    def apply_production(self, settings):
        settings['TVM_SERVICE_ID'] = 2002610

        notifications_db = settings['MONGO_DATABASES'][SUBURBAN_NOTIFICATION_DATABASE_NAME]
        notifications_db['password'] = get_secret('rasp-infocenter-production.SUBURBAN_NOTIFICATION_MONGO_PASSWORD')
        notifications_db['dbaas_id'] = 'mdbcntia2artsrqkam7o'
        notifications_db['fallback_hosts'] = [
            'vla-z9lxahss50j1r0bk.db.yandex.net',
            'man-m12mw64t0f4ind67.db.yandex.net',
            'sas-2r9yhkeww5wvsv33.db.yandex.net'
        ]

        configure_raven(settings, 'info_center')


Configuration.configurator_classes.append(InfoCenterConfigurator)
