# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import requests

from blackbox import XmlBlackbox
from travel.rasp.infra.api.base.client import get_retriable_session
from travel.library.python.tvm_ticket_provider import provider_fabric


BLACKBOX_RETRY_COUNT = 1
BLACKBOX_TIMEOUT = 0.05


class ApiException(Exception):
    pass


class ApiBase(object):
    def __init__(
        self, oauth_token=None,
        tvm_token=None, tvm_src=None, tvm_dst=None,
        blackbox_url=None, blackbox_tvm=None, ip=None
    ):
        self._session = get_retriable_session()
        self.tvm_token = tvm_token
        self.tvm_src = tvm_src
        self.tvm_dst = tvm_dst
        self.blackbox_url = blackbox_url
        self.blackbox_tvm = blackbox_tvm
        self.ip = ip
        self.authorization = 'OAuth {}'.format(oauth_token)

        if oauth_token is not None:
            self._session.headers.update({'Authorization': self.authorization})

        if self.tvm_token and self.tvm_src and self.tvm_dst:
            tvm_dsts = [self.tvm_dst]
            if self.blackbox_tvm:
                tvm_dsts.append(blackbox_tvm)

            self.tvm_provider = provider_fabric.create(
                source_id=self.tvm_src,
                destinations=tvm_dsts,
                secret=self.tvm_token
            )
        else:
            self.tvm_provider = None

        if self.blackbox_url and self.blackbox_tvm and self.ip:
            self.blackbox_client = XmlBlackbox(
                url=blackbox_url,
                retry_count=BLACKBOX_RETRY_COUNT,
                timeout=BLACKBOX_TIMEOUT,
            )
        else:
            self.blackbox_client = None

    def _get_user_ticket(self, tvm_ticket):
        response = self.blackbox_client.oauth(
            headers_or_token={
                'Authorization': self.authorization,
                'X-Ya-Service-Ticket': tvm_ticket
            },
            userip=self.ip,
            get_user_ticket='yes'
        )

        return response['user_ticket']

    def _make_request(self, method, *args, **kwargs):
        if self.tvm_provider:
            tvm_ticket = self.tvm_provider.get_ticket(self.tvm_dst)
            if tvm_ticket:
                self._session.headers.update({'X-Ya-Service-Ticket': tvm_ticket})

                if self.blackbox_client:
                    tvm_ticket = self.tvm_provider.get_ticket(self.blackbox_tvm)
                    user_ticket = self._get_user_ticket(tvm_ticket)
                    self._session.headers.update({'X-Ya-User-Ticket': user_ticket})

        logging.info('Make request: method=[%s], args=[%r], kwargs=[%r]', method, args, kwargs)
        response = getattr(self._session, method)(*args, **kwargs)
        try:
            response.raise_for_status()
        except requests.HTTPError:
            logging.exception('Api Error')
            logging.error('Response body: %s', response.text)
            raise

        return response

    def _make_json_request(self, method, *args, **kwargs):
        return self._make_request(method, *args, **kwargs).json()

    def _make_raw_request(self, method, *args, **kwargs):
        return self._make_request(method, *args, **kwargs).text
