import copy
import logging

from travel.rasp.infra.api.solomon import SolomonApi


DEFAULT_PERIOD = 60000
DEFAULT_DELAY = 10
DEFAULT_STATUS_CHECK_THRESHOLD = 300
DEFAULT_TIMINGS_CHECK_THRESHOLD = 500
DEFAULT_TEMPLATE = {
    'id': None,
    'projectId': None,
    'name': None,
    'createdBy': None,
    'updatedBy': None,
    'groupByLabels': [
        'host'
    ],
    'notificationChannels': [
        'dorblu-like-juggler'
    ],
    'type': {
        'expression': {
            'program': None,
            'checkExpression': None
        }
    },
    'annotations': {
        # https://st.yandex-team.ru/SOLOMON-3167
        'trafficLight.color': '{{expression.trafficColor}}'
    },
    'periodMillis': DEFAULT_PERIOD,
    'delaySeconds': DEFAULT_DELAY
}

CHECK_PROGRAM_HEAD = (
    "let is_crit = false;\n"
    "let is_warn = false;\n\n"
)
CHECK_PROGRAM_TAIL = "let trafficColor = is_crit ? 'red' : (is_warn ? 'yellow' : 'green');\n"

STATUS_CHECK_NGINX_LOGS_PROGRAM = (
    "let countFiltered = group_lines('sum', {{project='{project}', cluster='{cluster}', service='nginx_logs', "
    "application='{application}', status='{status}', sensor='count_with_status_aggregation', "
    "environment='{environment}', component='{component}', method='any', uri='{uri}', host='*'}});\n"
    "let countAll = group_lines('sum', {{project='{project}', cluster='{cluster}', service='nginx_logs', "
    "application='{application}', status='ANY', sensor='count_with_status_aggregation', environment='{environment}', "
    "component='{component}', method='any', uri='{uri}', host='*'}});\n"
    "let percent = avg(countFiltered / countAll);\n"
    "let is_crit = is_crit || percent > {crit_threshold};\n"
    "let is_warn = is_warn || percent > {warn_threshold};\n\n"
)
STATUS_CHECK_NGINX_LOGS_EXPRESSION = 'is_warn'

STATUS_CHECK_HTTP_PROGRAM = (
    "let countFiltered = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', http_code='{http_code}', sensor='http.http_code_count', "
    "environment='{environment}', component='{component}', endpoint='{endpoint}', host='*'}});\n"
    "let countAll = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', sensor='http.request_count', environment='{environment}', "
    "component='{component}', endpoint='{endpoint}', host='*'}});\n"
    "let percent = avg(countFiltered / countAll);\n"
    "let is_crit = is_crit || percent > {crit_threshold};\n"
    "let is_warn = is_warn || percent > {warn_threshold};\n\n"
)
STATUS_CHECK_HTTP_EXPRESSION = 'is_crit'

TIMINGS_CHECK_PROGRAM = (
    "let histogram = histogram_percentile({percentile}, 'bin', {{project='{project}', cluster='{cluster}', "
    "service='{service}', environment='{environment}', component='{component}', "
    "application='{application}', host='*', sensor='{sensor}'}});\n"
    "let is_crit = is_crit || sum(transform(tail(histogram, 5) - {crit_threshold}, 'heaviside')) >= 3;\n"
    "let is_warn = is_warn || sum(transform(tail(histogram, 5) - {warn_threshold}, 'heaviside')) >= 3;\n"
)
TIMINGS_CHECK_EXPRESSION = 'is_warn'

GRPC_ERRORS_PROGRAM = (
    "let countFiltered = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', sensor='grpc.errors_count', environment='{environment}', "
    "component='{component}', endpoint='{endpoint}', host='*'}});\n"
    "let countAll = group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', sensor='grpc.request_count', environment='{environment}', "
    "component='{component}', endpoint='{endpoint}', host='*'}});\n"
    "let percent = avg(countFiltered / countAll);\n"
    "let is_crit = is_crit || percent > {crit_threshold};\n"
    "let is_warn = is_warn || percent > {warn_threshold};\n\n"
)
GRPC_ERRORS_EXPRESSION = 'is_crit'

GO_PANICS_PROGRAM = (
    "let countPanics = sum(group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', sensor='{sensor}', environment='{environment}', "
    "component='{component}', endpoint='{endpoint}', host='*'}}));\n"
    "let is_crit = is_crit || countPanics > {crit_threshold};\n"
    "let is_warn = is_warn || countPanics > {warn_threshold};\n\n"
)
GO_PANICS_EXPRESSION = 'is_crit'

APP_ERRORS_PROGRAM = (
    "let countErrors = sum(group_lines('sum', {{project='{project}', cluster='{cluster}', service='{service}', "
    "application='{application}', sensor='{sensor}', environment='{environment}', "
    "component='{component}'}}));\n"
    "let is_crit = is_crit || countErrors > {crit_threshold};\n"
    "let is_warn = is_warn || countErrors > {warn_threshold};\n\n"
)
APP_ERRORS_EXPRESSION = 'is_crit'

MONGO_IS_ALIVE_PROGRAM = (
    "let is_alive = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', host='*', "
    "name='mongod-is_alive', node='by_host'}});\n"
    "let is_crit = is_crit || any_of(map(is_alive, x -> last(x) < 1));\n"
    "let is_crit = is_crit || count(is_alive) == 0;\n\n"
)

MONGO_IS_ALIVE_EXPRESSION = 'is_crit'

MONGO_FREE_SPACE_PROGRAM = (
    "let free_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-free_bytes_/var/lib/mongodb'}});\n"
    "let total_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-total_bytes_/var/lib/mongodb'}});\n\n"
    "let free_percent = last((free_bytes / total_bytes) * 100);\n"
    "let is_crit = is_crit || free_percent <= 10;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || free_percent <= 20;\n"
    "warn_if(is_warn);\n\n"
)

MONGO_FREE_SPACE_EXPRESSION = 'is_crit'

MONGO_CPU_PROGRAM = (
    "let cpu_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_usage', host='*.db.yandex.net'}};\n"
    "let cpu_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_limit', host='*.db.yandex.net'}};\n\n"
    "let cpu_percent = max(map(cpu_usage / cpu_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || cpu_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || cpu_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

MONGO_CPU_EXPRESSION = 'is_crit'

MONGO_MEMORY_PROGRAM = (
    "let memory_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_usage', host='*.db.yandex.net'}};\n"
    "let memory_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_limit', host='*.db.yandex.net'}};\n"
    "let cache_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cache_usage', host='*.db.yandex.net'}};\n\n"
    "let memory_percent = max(map((memory_usage - cache_usage) / memory_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || memory_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || memory_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

MONGO_MEMORY_EXPRESSION = 'is_crit'

REDIS_IS_ALIVE_PROGRAM = (
    "let is_alive = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', host='*', "
    "name='redis_is_alive', node='by_host'}});\n"
    "let is_crit = is_crit || any_of(map(is_alive, x -> last(x) < 1));\n"
    "let is_crit = is_crit || count(is_alive) == 0;\n\n"
)

REDIS_IS_ALIVE_EXPRESSION = 'is_crit'

REDIS_FREE_SPACE_PROGRAM = (
    "let free_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-free_bytes_/var/lib/redis'}});\n"
    "let total_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-total_bytes_/var/lib/redis'}});\n\n"
    "let free_percent = last((free_bytes / total_bytes) * 100);\n"
    "let is_crit = is_crit || free_percent <= 10;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || free_percent <= 20;\n"
    "warn_if(is_warn);\n\n"
)

REDIS_FREE_SPACE_EXPRESSION = 'is_crit'

REDIS_CPU_PROGRAM = (
    "let cpu_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_usage', host='*.db.yandex.net'}};\n"
    "let cpu_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_limit', host='*.db.yandex.net'}};\n\n"
    "let cpu_percent = max(map(cpu_usage / cpu_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || cpu_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || cpu_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

REDIS_CPU_EXPRESSION = 'is_crit'

REDIS_MEMORY_PROGRAM = (
    "let memory_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_usage', host='*.db.yandex.net'}};\n"
    "let memory_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_limit', host='*.db.yandex.net'}};\n"
    "let cache_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cache_usage', host='*.db.yandex.net'}};\n\n"
    "let memory_percent = max(map((memory_usage - cache_usage) / memory_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || memory_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || memory_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

REDIS_MEMORY_EXPRESSION = 'is_crit'

MYSQL_IS_ALIVE_PROGRAM = (
    "let is_alive = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', host='*', "
    "name='mysql_is_alive', node='by_host'}});\n"
    "let is_crit = is_crit || any_of(map(is_alive, x -> last(x) < 1));\n"
    "let is_crit = is_crit || count(is_alive) == 0;\n\n"
)

MYSQL_IS_ALIVE_EXPRESSION = 'is_crit'

MYSQL_PRIMARY_CHANGED_PROGRAM = """
let is_primary = series_sum({{project="internal-mdb", cluster="{cluster}", service="mdb", host="*", name="mysql_is_primary", node="by_host"}});

// Если avg > 0, то хост был недавно мастером. И если последнее значение == 0, значит он перестал быть мастером
let not_primary_anymore = any_of(
  map(
    is_primary,
    p -> avg(p) > 0 && last(p) == 0
  )
);
let is_crit = not_primary_anymore;
"""
MYSQL_PRIMARY_CHANGED_EXPRESSION = 'is_crit'

MYSQL_FREE_SPACE_PROGRAM = (
    "let free_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-free_bytes_/var/lib/mysql'}});\n"
    "let total_bytes = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='disk-total_bytes_/var/lib/mysql'}});\n\n"
    "let free_percent = last((free_bytes / total_bytes) * 100);\n"
    "let is_crit = is_crit || free_percent <= 10;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || free_percent <= 20;\n"
    "warn_if(is_warn);\n\n"
)

MYSQL_FREE_SPACE_EXPRESSION = 'is_crit'

MYSQL_REPLICATION_LAG_PROGRAM = (
    "let replication_lag = series_sum({{project='internal-mdb', cluster='{cluster}', service='mdb', node='by_host', "
    "host='*', name='mysql_replication_lag'}});\n"
    "let current_lag = last(replication_lag);\n\n"
    "let is_crit = is_crit || current_lag >= 60;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || current_lag >= 5;\n"
    "warn_if(is_warn);\n\n"
)

MYSQL_REPLICATION_LAG_EXPRESSION = 'is_crit'

MYSQL_CPU_PROGRAM = (
    "let cpu_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_usage', host='*.db.yandex.net'}};\n"
    "let cpu_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cpu_limit', host='*.db.yandex.net'}};\n\n"
    "let cpu_percent = max(map(cpu_usage / cpu_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || cpu_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || cpu_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

MYSQL_CPU_EXPRESSION = 'is_crit'

MYSQL_MEMORY_PROGRAM = (
    "let memory_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_usage', host='*.db.yandex.net'}};\n"
    "let memory_limit = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/memory_limit', host='*.db.yandex.net'}};\n"
    "let cache_usage = {{project='internal-mdb', cluster='internal-mdb_dom0', service='dom0', "
    "container='{container}', sensor='/porto/cache_usage', host='*.db.yandex.net'}};\n\n"
    "let memory_percent = max(map((memory_usage - cache_usage) / memory_limit, x->last(x) * 100));\n\n"
    "let is_crit = is_crit || memory_percent >= 90;\n"
    "alarm_if(is_crit);\n\n"
    "let is_warn = is_warn || memory_percent >= 80;\n"
    "warn_if(is_warn);\n\n"
)

MYSQL_MEMORY_EXPRESSION = 'is_crit'


class SolomonAlertUpdater(object):
    @staticmethod
    def _update_thresholds(thresholds, default):
        if 'crit' not in thresholds:
            thresholds['crit'] = default
        if 'warn' not in thresholds:
            thresholds['warn'] = thresholds['crit']

    def status_check_nginx_logs_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_STATUS_CHECK_THRESHOLD)
            program += STATUS_CHECK_NGINX_LOGS_PROGRAM.format(
                project=self.project,
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                status=alert_info.get('status', '5xx'),
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                uri=check_config.get('uri', '*'),
            )
        return program + CHECK_PROGRAM_TAIL

    def status_check_http_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_STATUS_CHECK_THRESHOLD)
            program += STATUS_CHECK_HTTP_PROGRAM.format(
                project=self.project,
                http_code=alert_info.get('status', '5XX').upper(),
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                endpoint=check_config.get('uri', '*'),
                service=alert_info.get('service', 'http')
            )
        return program + CHECK_PROGRAM_TAIL

    def timings_check_program(self, alert_info, service, sensor):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_TIMINGS_CHECK_THRESHOLD)
            program += TIMINGS_CHECK_PROGRAM.format(
                project=self.project,
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                percentile=check_config.get('percentile', '95.0'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                service=service,
                sensor=sensor,
            )
        return program + CHECK_PROGRAM_TAIL

    def grpc_errors_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_STATUS_CHECK_THRESHOLD)
            program += GRPC_ERRORS_PROGRAM.format(
                project=self.project,
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                endpoint=check_config.get('uri', '*'),
                service=alert_info.get('service', 'grpc')
            )
        return program + CHECK_PROGRAM_TAIL

    def go_panics_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_STATUS_CHECK_THRESHOLD)
            program += GO_PANICS_PROGRAM.format(
                project=self.project,
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                endpoint=check_config.get('uri', '*'),
                service=alert_info['service'],
                sensor=alert_info['sensor'],
            )
        return program + CHECK_PROGRAM_TAIL

    def app_errors_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        for check_config in alert_info.get('checks', []):
            self._update_thresholds(check_config, DEFAULT_STATUS_CHECK_THRESHOLD)
            program += APP_ERRORS_PROGRAM.format(
                project=self.project,
                cluster=alert_info.get('cluster', 'production'),
                application=alert_info['application'],
                environment=alert_info.get('environment', 'production'),
                component=alert_info.get('component', 'main'),
                crit_threshold=check_config['crit'],
                warn_threshold=check_config['warn'],
                service=alert_info['service'],
                sensor=alert_info['sensor'],
            )
        return program + CHECK_PROGRAM_TAIL

    def mongo_is_alive_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += MONGO_IS_ALIVE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def mongo_free_space_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += MONGO_FREE_SPACE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def mongo_cpu_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += MONGO_CPU_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def mongo_memory_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += MONGO_MEMORY_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def redis_is_alive_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += REDIS_IS_ALIVE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def redis_free_space_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += REDIS_FREE_SPACE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def redis_cpu_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += REDIS_CPU_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def redis_memory_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += REDIS_MEMORY_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_is_alive_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += MYSQL_IS_ALIVE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_free_space_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += MYSQL_FREE_SPACE_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_replication_lag_program(self, alert_info):
        program = CHECK_PROGRAM_HEAD
        program += MYSQL_REPLICATION_LAG_PROGRAM.format(
            cluster=alert_info['cluster']
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_cpu_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += MYSQL_CPU_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_memory_program(self, alert_info):
        container = '|'.join(alert_info['container'])
        program = CHECK_PROGRAM_HEAD
        program += MYSQL_MEMORY_PROGRAM.format(
            container=container
        )
        return program + CHECK_PROGRAM_TAIL

    def mysql_primary_changed_program(self, alert_info):
        program = MYSQL_PRIMARY_CHANGED_PROGRAM.format(cluster=alert_info['cluster'])
        return CHECK_PROGRAM_HEAD + program + CHECK_PROGRAM_TAIL

    def prepare_alert_config(self, alert_id, alert_info):
        template = copy.deepcopy(DEFAULT_TEMPLATE)
        template['projectId'] = self.project
        template['createdBy'], template['updatedBy'] = self.alert_owner, self.alert_owner
        additional_params = alert_info.get('additional_params', {})
        template.update(additional_params)

        template['id'] = alert_id
        template['name'] = alert_info.get('name', alert_id)

        if 'notificationChannels' in alert_info:
            template['notificationChannels'] = alert_info['notificationChannels']
        if 'groupByLabels' in alert_info:
            template['groupByLabels'] = alert_info['groupByLabels']
        template['periodMillis'] = int(alert_info.get('periodMillis', DEFAULT_PERIOD))
        template['delaySeconds'] = alert_info.get('delaySeconds', DEFAULT_DELAY)
        template['annotations'].update(alert_info.get('annotations', {}))

        alert_type = alert_info.get('type', '').lower()
        if alert_type == 'status_check':
            template['type']['expression']['program'] = self.status_check_nginx_logs_program(alert_info)
            template['type']['expression']['checkExpression'] = STATUS_CHECK_NGINX_LOGS_EXPRESSION
        elif alert_type == 'status_check_http':
            template['type']['expression']['program'] = self.status_check_http_program(alert_info)
            template['type']['expression']['checkExpression'] = STATUS_CHECK_HTTP_EXPRESSION
        elif alert_type == 'timings':
            template['type']['expression']['program'] = self.timings_check_program(
                alert_info, alert_info.get('service', 'nginx_logs'), 'timings_hist'
            )
            template['type']['expression']['checkExpression'] = TIMINGS_CHECK_EXPRESSION
        elif alert_type == 'timings_http':
            template['type']['expression']['program'] = self.timings_check_program(
                alert_info, alert_info.get('service', 'http'), 'http.request_duration'
            )
            template['type']['expression']['checkExpression'] = TIMINGS_CHECK_EXPRESSION
        elif alert_type == 'timings_grpc':
            template['type']['expression']['program'] = self.timings_check_program(
                alert_info, alert_info.get('service', 'grpc'), 'grpc.request_duration'
            )
            template['type']['expression']['checkExpression'] = TIMINGS_CHECK_EXPRESSION
        elif alert_type == 'grpc_errors':
            template['type']['expression']['program'] = self.grpc_errors_program(alert_info)
            template['type']['expression']['checkExpression'] = GRPC_ERRORS_EXPRESSION
        elif alert_type == 'go_panics':
            template['type']['expression']['program'] = self.go_panics_program(alert_info)
            template['type']['expression']['checkExpression'] = GO_PANICS_EXPRESSION
        elif alert_type == 'app_errors':
            template['type']['expression']['program'] = self.app_errors_program(alert_info)
            template['type']['expression']['checkExpression'] = APP_ERRORS_EXPRESSION
        elif alert_type == 'mongo_is_alive':
            template['type']['expression']['program'] = self.mongo_is_alive_program(alert_info)
            template['type']['expression']['checkExpression'] = MONGO_IS_ALIVE_EXPRESSION
        elif alert_type == 'mongo_free_space':
            template['type']['expression']['program'] = self.mongo_free_space_program(alert_info)
            template['type']['expression']['checkExpression'] = MONGO_FREE_SPACE_EXPRESSION
        elif alert_type == 'mongo_cpu':
            template['type']['expression']['program'] = self.mongo_cpu_program(alert_info)
            template['type']['expression']['checkExpression'] = MONGO_CPU_EXPRESSION
        elif alert_type == 'mongo_memory':
            template['type']['expression']['program'] = self.mongo_memory_program(alert_info)
            template['type']['expression']['checkExpression'] = MONGO_MEMORY_EXPRESSION
        elif alert_type == 'redis_is_alive':
            template['type']['expression']['program'] = self.redis_is_alive_program(alert_info)
            template['type']['expression']['checkExpression'] = REDIS_IS_ALIVE_EXPRESSION
        elif alert_type == 'redis_free_space':
            template['type']['expression']['program'] = self.redis_free_space_program(alert_info)
            template['type']['expression']['checkExpression'] = REDIS_FREE_SPACE_EXPRESSION
        elif alert_type == 'redis_cpu':
            template['type']['expression']['program'] = self.redis_cpu_program(alert_info)
            template['type']['expression']['checkExpression'] = REDIS_CPU_EXPRESSION
        elif alert_type == 'redis_memory':
            template['type']['expression']['program'] = self.redis_memory_program(alert_info)
            template['type']['expression']['checkExpression'] = REDIS_MEMORY_EXPRESSION
        elif alert_type == 'mysql_is_alive':
            template['type']['expression']['program'] = self.mysql_is_alive_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_IS_ALIVE_EXPRESSION
        elif alert_type == 'mysql_primary_changed':
            template['type']['expression']['program'] = self.mysql_primary_changed_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_PRIMARY_CHANGED_EXPRESSION
        elif alert_type == 'mysql_free_space':
            template['type']['expression']['program'] = self.mysql_free_space_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_FREE_SPACE_EXPRESSION
        elif alert_type == 'mysql_replication_lag':
            template['type']['expression']['program'] = self.mysql_replication_lag_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_REPLICATION_LAG_EXPRESSION
        elif alert_type == 'mysql_cpu':
            template['type']['expression']['program'] = self.mysql_cpu_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_CPU_EXPRESSION
        elif alert_type == 'mysql_memory':
            template['type']['expression']['program'] = self.mysql_memory_program(alert_info)
            template['type']['expression']['checkExpression'] = MYSQL_MEMORY_EXPRESSION
        elif alert_type != 'custom':
            raise ValueError('Wrong alert type: {}'.format(alert_info.get('type')))

        return template

    def delete_old_alerts(self, api, new_alerts, old_alerts):
        logging.info('Deleting old alerts...')
        logging.debug(old_alerts)
        for old_alert_id in old_alerts:
            if old_alert_id not in new_alerts:
                logging.debug('Deleting {} alert'.format(old_alert_id))
                old_alert_config = api.get_alert(old_alert_id)
                if 'protected' in old_alert_config.get('annotations', {}):
                    logging.debug('Skipping {} alert'.format(old_alert_id))
                    continue
                api.delete_alert(old_alert_id)
                logging.debug(old_alert_config)
        logging.info('Successfully deleted old alerts')

    def update_new_alerts(self, api, new_alerts, old_alerts):
        logging.info('Updating alerts...')
        for new_alert_id, new_alert_data in new_alerts.items():
            alert_config = self.prepare_alert_config(new_alert_id, new_alert_data)
            if new_alert_id in old_alerts:
                logging.debug('Updating alert {} with config: {}'.format(new_alert_id, alert_config))
                alert_config['version'] = api.get_alert(new_alert_id)['version']
                api.update_alert(new_alert_id, alert_config)
            else:
                logging.debug('Creating new alert with config: {}'.format(alert_config))
                api.create_alert(alert_config)
        logging.info('Successfully updated alerts')

    def run(self):
        new_alerts = self.config
        logging.debug(new_alerts)
        api = SolomonApi(project=self.project, oauth_token=self.solomon_token)
        current_alerts = api.get_active_alerts_ids()
        self.delete_old_alerts(api, new_alerts, current_alerts)
        self.update_new_alerts(api, new_alerts, current_alerts)

    def __init__(self, project, config, solomon_token, alert_owner):
        self.project = project
        self.config = config
        self.solomon_token = solomon_token
        self.alert_owner = alert_owner
