# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import tarfile
from retry import retry

from travel.rasp.library.python.shell_utils import logged_check_call, cd


@retry(tries=3, delay=3)
def _upload_to_s3(aws_url, aws_bucket, app_prefix, sync_dir):
    logged_check_call('aws s3 --endpoint-url={aws_url} sync {sync_dir} s3://{aws_bucket}/{app_prefix}/'.format(
        aws_url=aws_url,
        aws_bucket=aws_bucket,
        app_prefix=app_prefix,
        sync_dir=sync_dir,
    ))


def build_markup(version):
    logged_check_call('npm config set prefix /usr/local', skip_on_err=True)
    logged_check_call('npm install -g bower --registry=http://npm.yandex-team.ru', skip_on_err=True)

    for directory in ('static', 'markup'):
        with cd(directory):
            logged_check_call('nvm use 0.10', skip_on_err=True)
            logged_check_call('YENV=production make > build.log')
            logged_check_call('rm -rf node_modules', skip_on_err=True)

    with open('markup-version', 'w') as f:
        f.write(version)


def pack_markup():
    with tarfile.open('markup.tar', 'w') as tar:
        tar.add('static')
        tar.add('markup')
        tar.add('markup-version')


def upload_media_to_s3(aws_url, aws_bucket, app_prefix):
    _upload_to_s3(aws_url, aws_bucket, app_prefix + '/old', './markup/old/')
    _upload_to_s3(aws_url, aws_bucket, app_prefix + '/new', './static/new/')


def upload_to_s3(aws_url, aws_bucket, app_prefix, version):
    _upload_to_s3(aws_url, aws_bucket, app_prefix + '/{}/static'.format(version), './static')
    _upload_to_s3(aws_url, aws_bucket, app_prefix + '/{}/markup'.format(version), './markup')
