#!/usr/bin/env python3

import json
import os
import random
from time import sleep

import requests

JUGGLER_URL = 'http://juggler-push.search.yandex.net/events'
JUGGLER_TIMEOUT = 5

IM_API_HOST = os.getenv('PARTNER_CHECK_IM_HOST', 'api.smarttravel.ru')
IM_API_URL = 'https://{host}/Order/V1/Info/OrderInfo'.format(host=IM_API_HOST)
IM_CRITICAL_ERROR_CODES = {11, 12, 13, 14}
IM_PARAMETER_ERROR_CODES = {41, 42, 43, 44, 48}
IM_TIMEOUT = 10

SECONDS_BETWEEN_CHECKS = 2 * 60
MAX_RANDOM_SLEEP_SECONDS = max(0, SECONDS_BETWEEN_CHECKS - IM_TIMEOUT - JUGGLER_TIMEOUT - 5)
NUMBER_OF_INSTANCES_WITH_CHECK = int(os.getenv('PARTNER_CHECK_NUMBER_OF_INSTANCES', 6))
SECONDS_BETWEEN_INSTANCES = SECONDS_BETWEEN_CHECKS / NUMBER_OF_INSTANCES_WITH_CHECK


def check_im():
    """
    Для проверки доступа делаем запрос OrderInfo без параметра и смотрим на код ошибки.
    Если ошибка связана с параметром, то доступ есть.
    https://st.yandex-team.ru/TRAINS-129#1534853851000
    """
    try:
        login = os.getenv('PARTNER_CHECK_IM_LOGIN')
        password = os.getenv('PARTNER_CHECK_IM_PASSWORD')
        resp = requests.post(
            IM_API_URL,
            auth=(login, password),
            headers={
                'POS': os.getenv('PARTNER_CHECK_IM_POS'),
                'Content-Type': 'application/json',
            },
            data=json.dumps({'OrderId': None}),
            timeout=IM_TIMEOUT
        )
        code = resp.json().get('Code')
        if code in IM_PARAMETER_ERROR_CODES:
            return 'OK'
        elif code in IM_CRITICAL_ERROR_CODES:
            return 'CRIT'
        return 'WARN'
    except Exception:
        return 'CRIT'


def send_check_result_to_juggler(status):
    if status not in ('OK', 'CRIT', 'WARN'):
        raise ValueError('Bad status. Expected OK, CRIT or WARN.')
    data = {
        'source': '{}.{}.{}.{}'.format(os.getenv('DEPLOY_STAGE_ID'), os.getenv('DEPLOY_UNIT_ID')),
        'events': [
            {
                'description': 'ICMP IM partner check',
                'host': os.getenv('DEPLOY_POD_PERSISTENT_FQDN'),
                'instance': '',
                'service': 'train-partner-im-ping',
                'status': status,
                'tags': [
                    'im',
                    'partner',
                ],
            },
        ],
    }
    requests.post(JUGGLER_URL, data=json.dumps(data), timeout=JUGGLER_TIMEOUT)


def wait_my_turn():
    """
    Нужно разнести запросы к партнеру по времени
    """
    try:
        instance_number = int(os.getenv('QLOUD_DISCOVERY_INSTANCE').split('.')[0].rsplit('-', 1)[1])
    except (AttributeError, IndexError, ValueError):
        # если не получилось разобрать, то просто спим рандомно
        sleep(random.uniform(0, MAX_RANDOM_SLEEP_SECONDS))
    else:
        sleep((instance_number - 1) * SECONDS_BETWEEN_INSTANCES)


if __name__ == '__main__':
    if os.getenv('PARTNER_CHECK_IS_ON'):
        wait_my_turn()
        send_check_result_to_juggler(check_im())
