# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import hashlib
import os
import random
import signal

import yatest
from travel.library.recipe_utils.utils import (
    is_alive, log, run_process, Timer, untar, wait_for_process, get_free_port, set_environ
)


description = 'mongodb-step'

_MONGODB_DIST_DIR = 'mongodb'


def run_mongodb(uniq):
    random.seed(uniq)
    mongo_port = get_free_port(desired_port=27017 + random.randrange(1000))
    log('Starting mongo on port:', mongo_port)

    mongodb_pid_file = 'mongod_{}.pid'.format(uniq)
    mongo_dir = os.path.join(os.getcwd(), uniq, _MONGODB_DIST_DIR)
    mongo_data_dir = os.path.join(mongo_dir, 'data')
    mongod_path = os.path.join(mongo_dir, 'bin', 'mongod')
    os.mkdir(mongo_data_dir)

    set_environ('RASP_MONGO_RECIPE_MONGODB_PID', mongodb_pid_file)
    set_environ('RASP_MONGO_RECIPE_DB_NAME_SUFFIX', '_{}_tests'.format(uniq))
    set_environ('RASP_MONGO_RECIPE_PORT', str(mongo_port))

    run_process([mongod_path, '--version'], 'mongo_version', os.environ.copy())

    with Timer('Starting mongodb...'):
        process_name = 'mongodb_{}'.format(uniq)
        process = run_process([
            mongod_path,
            '--nounixsocket',
            '--pidfilepath', mongodb_pid_file,
            '--dbpath', mongo_data_dir,
            '--port', mongo_port,
            '--bind_ip', 'localhost',
            '--logpath', '{}/mongo.log'.format(mongo_dir),
            '--ipv6',
        ], process_name, os.environ.copy())

        wait_for_process(process, process_name, lambda: is_alive(mongo_port, process_name))


def setup():
    uniq = hashlib.md5(os.getcwd().encode("utf-8")).hexdigest()

    mongo_dir = os.path.join(os.getcwd(), uniq)
    try:
        os.makedirs(mongo_dir)
    except OSError:
        pass

    mongo_archive_path = yatest.common.build_path('travel/rasp/library/common_recipe/packages/mongodb/mongodb.tgz')

    untar(mongo_archive_path, path=mongo_dir)
    run_mongodb(uniq)


def teardown():
    pid_file = os.getenv('RASP_MONGO_RECIPE_MONGODB_PID', 'mongod.pid')
    with open(pid_file, 'r') as f:
        pid = int(f.read())
        os.kill(pid, signal.SIGTERM)
