package ru.yandex.travel.bus.service;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.util.HttpConstants;
import org.springframework.http.HttpStatus;

import ru.yandex.travel.bus.model.BusPointsHttp;
import ru.yandex.travel.bus.model.BusRideHttp;
import ru.yandex.travel.bus.model.BusSearchHttpResponse;
import ru.yandex.travel.bus.model.BusCalendarDay;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;

@Slf4j
public class BusesHttpApiService {
    private final AsyncHttpClientWrapper ahcClient;
    private final BusesHttpApiServiceProperties config;

    public BusesHttpApiService(AsyncHttpClientWrapper ahcClient,
                               BusesHttpApiServiceProperties config) {
        this.ahcClient = ahcClient;
        this.config = config;
    }

    public CompletableFuture<Map<String, BusCalendarDay>> calendar(String fromKey, String toKey, String maxDate, String requestId) {
        String url = config.getBaseUrl() + config.getCalendarPath();

        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(url)
                .addQueryParam("from-id", fromKey)
                .addQueryParam("to-id", toKey)
                .addQueryParam("max-date", maxDate);

        return ahcClient.executeRequest(requestBuilder, "busCalendar", requestId).thenApply(this::parseCalendarResponse);
    }

    private Map<String, BusCalendarDay> parseCalendarResponse(Response response) {
        if (response.getStatusCode() != HttpStatus.OK.value()) {
            log.error("BusesHttpApiService::parseSearchResponse returned non-200 status code: {}", response.getStatusCode());
            throw new RuntimeException("Bad backend response code");
        }
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            return objectMapper.readValue(response.getResponseBody(), new TypeReference<Map<String,BusCalendarDay>>() {});
        } catch (JsonProcessingException e) {
            log.error("BusesHttpApiService::parseSearchResponse bad response: ", e);
            throw new RuntimeException("Bad backend response body");
        }
    }

    public CompletableFuture<BusSearchHttpResponse> search(String fromKey, String toKey, String date, String requestId) {
        String url = config.getBaseUrl() + config.getSearchPath();

        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(url)
                .addQueryParam("from-id", fromKey)
                .addQueryParam("to-id", toKey)
                .addQueryParam("v", "2")
                .addQueryParam("date", date);

        return ahcClient.executeRequest(requestBuilder, "busSearch", requestId).thenApply(this::parseSearchResponse);
    }

    private BusSearchHttpResponse parseSearchResponse(Response response) {
        if (response.getStatusCode() != HttpStatus.OK.value() && response.getStatusCode() != HttpStatus.ACCEPTED.value()) {
            log.error("BusesHttpApiService::parseSearchResponse returned non-200 status code: {}", response.getStatusCode());
            throw new RuntimeException("Bad backend response code");
        }
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            BusRideHttp[] busRides = objectMapper.readValue(response.getResponseBody(), BusRideHttp[].class);
            objectMapper.configure(DeserializationFeature.USE_JAVA_ARRAY_FOR_JSON_ARRAY, true);
            boolean isPartialResult = response.getStatusCode() == HttpStatus.ACCEPTED.value() ? true : false;
            return new BusSearchHttpResponse(busRides, isPartialResult);
        } catch (JsonProcessingException e) {
            log.error("BusesHttpApiService::parseSearchResponse bad response: ", e);
            throw new RuntimeException("Bad backend response body");
        }
    }

    public CompletableFuture<BusPointsHttp> points(String where, String requestId) {
        String url = config.getBaseUrl() + config.getPointsPath();

        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(url)
                .addQueryParam("from", "serp-resolve")
                .addQueryParam("where", where);

        return ahcClient.executeRequest(requestBuilder, "busPoints", requestId).thenApply(this::parsePointsResponse);
    }

    private BusPointsHttp parsePointsResponse(Response response) {
        if (response.getStatusCode() != HttpStatus.OK.value()) {
            log.error("BusesHttpApiService::parsePointsResponse returned non-200 status code: {}", response.getStatusCode());
            throw new RuntimeException("Bad backend response code");
        }
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            return objectMapper.readValue(response.getResponseBody(), BusPointsHttp.class);
        } catch (JsonProcessingException e) {
            log.error("BusesHttpApiService::parseSearchResponse bad response: ", e);
            throw new RuntimeException("Bad backend response body");
        }
    }
}
