package ru.yandex.travel.bus.service;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.travel.bus.model.BusDocumentType;
import ru.yandex.travel.bus.model.BusGenderType;
import ru.yandex.travel.bus.model.BusOrderStatus;
import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusTicketType;
import ru.yandex.travel.bus.model.BusVatType;
import ru.yandex.travel.bus.model.BusesOrder;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.buses.backend.proto.ETicketType;
import ru.yandex.travel.buses.backend.proto.TCitizenship;
import ru.yandex.travel.buses.backend.proto.TDocumentType;
import ru.yandex.travel.buses.backend.proto.TGender;
import ru.yandex.travel.buses.backend.proto.TSeat;
import ru.yandex.travel.buses.backend.proto.worker.TBookPassenger;
import ru.yandex.travel.buses.backend.proto.worker.TBookPassengerTicketType;
import ru.yandex.travel.buses.backend.proto.worker.TOrder;
import ru.yandex.travel.buses.backend.proto.worker.TTicket;
import ru.yandex.travel.buses.backend.proto.worker.TTicketPassenger;
import ru.yandex.travel.commons.proto.ProtoUtils;

public class BusesModelsHelpers {
    public static BusesPassenger tPassengerToBusesPassenger(TTicketPassenger proto) {
        BusesPassenger passenger = new BusesPassenger();

        passenger.setFirstName(proto.getFirstName());
        if (proto.hasMiddleName()) {
            passenger.setMiddleName(proto.getMiddleName());
        }
        passenger.setLastName(proto.getLastName());
        if (proto.hasBirthDate()) {
            passenger.setBirthday(ProtoUtils.toLocalDate(proto.getBirthDate()));
        }
        if (proto.hasGender()) {
            passenger.setGender(BusGenderType.BY_PROTO.getByValue(proto.getGender()));
        }
        if (proto.hasDocumentType()) {
            passenger.setDocumentType(BusDocumentType.BY_PROTO.getByValue(proto.getDocumentType()));
        }
        if (proto.hasDocumentNumber()) {
            passenger.setDocumentNumber(proto.getDocumentNumber());
        }
        if (proto.getTicketType() != ETicketType.TICKET_TYPE_UNKNOWN) {
            passenger.setTicketType(BusTicketType.BY_PROTO.getByValue(proto.getTicketType()));
        }
        if (!proto.getSeat().isEmpty()) {
            passenger.setSeatId(proto.getSeat());
        }
        if (proto.hasCitizenship()) {
            passenger.setCitizenship(proto.getCitizenship());
        }
        return passenger;
    }

    public static TBookPassenger busesPassengerToTPassenger(BusesPassenger passenger) {
        var p = TBookPassenger
                .newBuilder()
                .setFirstName(passenger.getFirstName())
                .setMiddleName(passenger.getMiddleName())
                .setLastName(passenger.getLastName())
                .setBirthDate(ProtoUtils.toTDate(passenger.getBirthday()))
                .setGender(TGender
                        .newBuilder()
                        .setId(passenger.getGender().getProtoValue())
                        .setPartnerId(passenger.getGenderPartnerId()))
                .setDocumentType(TDocumentType
                        .newBuilder()
                        .setId(passenger.getDocumentType().getProtoValue())
                        .setPartnerId(passenger.getDocumentTypePartnerId()))
                .setDocumentNumber(passenger.getDocumentNumber())
                .setTicketType(TBookPassengerTicketType
                        .newBuilder()
                        .setId(passenger.getTicketType().getProtoValue())
                        .setPartnerId(passenger.getTicketTypePartnerId()))
                .setCitizenship(TCitizenship.newBuilder()
                        .setId(passenger.getCitizenship())
                        .setPartnerId(passenger.getCitizenshipPartnerId())
                        .build());
        if (!Strings.isNullOrEmpty(passenger.getSeatId())) {
            p.setSeat(TSeat.newBuilder().setId(passenger.getSeatId()));
        }
        if (!Strings.isNullOrEmpty(passenger.getSeatPartnerId())) {
            p.getSeatBuilder().setPartnerId(passenger.getSeatPartnerId());
        }
        return p.build();
    }

    public static BusesTicket tTicketToBusesTicket(TTicket proto, Map<ETicketType, Money> feeMap) {
        BusesTicket ticket = new BusesTicket();
        ticket.setId(proto.getId());
        mergeConfirmedTicketFields(proto, ticket);
        ticket.setPrice(ProtoUtils.fromTPrice(proto.getPrice()));
        if (ProtoUtils.hasFields(proto.getRevenue())) { // TODO: remove this after buses-backend release with BUSES-1594
            ticket.setRevenue(ProtoUtils.fromTPrice(proto.getRevenue()));
        }
        ticket.setPartnerFee(feeMap.getOrDefault(proto.getPassenger().getTicketType(), Money.zero(ticket.getPrice().getCurrency())));
        ticket.setTicketVat(BusVatType.BY_PROTO.getByValue(proto.getPriceVat()));
        ticket.setPartnerFeeVat(BusVatType.BY_PROTO.getByValue(proto.getFeeVat()));
        if (proto.hasPassenger()) {
            ticket.setPassenger(BusesModelsHelpers.tPassengerToBusesPassenger(proto.getPassenger()));
        }
        return ticket;
    }

    public static void mergeConfirmedTicketFields(TTicket source, BusesTicket target) {
        target.setStatus(BusTicketStatus.BY_PROTO.getByValue(source.getStatus()));
        target.setBlankUrl(source.getBlankUrl());
        target.setCode(source.getCode());
        target.setSeries(source.getSeries());
        target.setNumber(source.getNumber());
        target.setBarcode(source.getBarcode());
        target.setPlatform(source.getPlatform());
    }

    public static BusesOrder tOrderToBusesOrder(TOrder proto, Map<ETicketType, Money> feeMap) {
        BusesOrder order = new BusesOrder();
        order.setId(proto.getId());
        order.setStatus(BusOrderStatus.BY_PROTO.getByValue(proto.getStatus()));
        order.setExpiresAt(ProtoUtils.toInstant(proto.getExpiresAt()));
        ArrayList<BusesTicket> tickets = new ArrayList<>();
        for (TTicket protoTicket : proto.getTicketsList()) {
            tickets.add(tTicketToBusesTicket(protoTicket, feeMap));
        }
        order.setTickets(tickets);
        return order;
    }

    public static void migrateTOrderToBusesOrder(TOrder source, BusesOrder target) {
        Preconditions.checkState(source.getId().equals(target.getId()),
                "Bus order migrate error. Order ids is not the same");
        Preconditions.checkState(source.getTicketsCount() == target.getTickets().size(),
                "Bus order migrate error. Tickets count is not the same");
        target.setStatus(BusOrderStatus.BY_PROTO.getByValue(source.getStatus()));
        Map<String, TTicket> sourceTickets = source.getTicketsList().stream()
                .collect(Collectors.toMap(TTicket::getId, x -> x));
        for (BusesTicket ticket : target.getTickets()) {
            TTicket sourceTicket = sourceTickets.get(ticket.getId());
            Preconditions.checkState(sourceTicket != null,
                    "Bus order migrate error. Tickets is not the same");
            mergeConfirmedTicketFields(sourceTicket, ticket);
        }
    }

    public static LocalDateTime toLocalTime(Instant dateTime, String zone) {
        if (dateTime == null) {
            return null;
        }
        ZoneId zoneId = ZoneId.of(zone);
        return dateTime.atZone(zoneId).toLocalDateTime();
    }
}
