package ru.yandex.travel.bus.service

import io.grpc.ManagedChannelBuilder
import io.grpc.health.v1.HealthCheckRequest
import io.grpc.health.v1.HealthCheckResponse
import io.grpc.health.v1.HealthGrpc
import io.opentracing.Tracer
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Primary
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.commons.network.NetworkUtils
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper
import ru.yandex.travel.orders.client.*
import ru.yandex.travel.tvm.TvmWrapper

@Configuration
@EnableConfigurationProperties(BusesServiceProperties::class)
open class BusesServiceConfiguration {
    @Bean
    open fun busesServiceStubFactory(
        @Qualifier("BusesGrpcChannelFactory") haGrpcChannelFactory: HAGrpcChannelFactory,
    ) = BusesServiceStubFactory(haGrpcChannelFactory)

    @Bean("BusesGrpcChannelFactory")
    open fun haGrpcChannelFactory(
        @Qualifier("BusesServiceProperties") properties: BusesServiceProperties,
        @Qualifier("BusesChannelSupplier") channelSupplier: ChannelSupplier,
        @Autowired(required = false) tvm: TvmWrapper?,
        @Autowired(required = false) tracer: Tracer?,
    ): HAGrpcChannelFactory {
        val tvmHelper = properties.tvm.let {
            if (it == null || !it.enabled || tvm == null) {
                return@let null
            }
            tvm.validateAlias(it.destinationAlias)
            return@let TvmHeaderClientHelper(tvm)
        }
        return HAGrpcChannelFactory.Builder.newBuilder()
            .withPingProducer { channel ->
                FutureUtils.buildCompletableFuture(
                    HealthGrpc
                        .newFutureStub(channel)
                        .check(HealthCheckRequest.newBuilder().build())
                ).thenApply { rsp ->
                    if (rsp.status == HealthCheckResponse.ServingStatus.SERVING) ChannelState.READY else ChannelState.NOT_READY
                }
            }
            .withFailureDetectorProperties(properties.failureDetection!!)
            .withChannelSupplier(channelSupplier)
            .withChannelBuilder { createChannel(it, properties, tvmHelper, tracer) }
            .build()
    }

    @Bean("BusesChannelSupplier")
    open fun getChannelSupplier(@Qualifier("BusesServiceProperties") properties: BusesServiceProperties): ChannelSupplier {
        val grpcChannelSupplierFactory = GrpcChannelSupplierFactory(properties)
        return grpcChannelSupplierFactory.channelSupplier
    }

    private fun createChannel(
        target: String,
        @Qualifier("BusesServiceProperties") properties: BusesServiceProperties,
        tvmHelper: TvmHeaderClientHelper?,
        tracer: Tracer?
    ): LabeledChannel {
        val interceptors = mutableListOf(
            LoggingClientInterceptor(
                NetworkUtils.getLocalHostName(), target, setOf(HealthGrpc.getCheckMethod().fullMethodName), tracer, true
            ),
            DefaultTimeoutClientInterceptor(properties.timeout),
            TracerClientInterceptor(tracer)
        )
        if (tvmHelper != null) {
            interceptors.add(tvmHelper.getInterceptor(properties.tvm!!.destinationAlias))
        }
        return LabeledChannel(
            target,
            ManagedChannelBuilder
                .forTarget(target)
                .intercept(interceptors)
                .usePlaintext()
                .build()
        )
    }

    @Bean
    @Primary
    open fun busesService(
        busesServiceStubFactory: BusesServiceStubFactory
    ): BusesService = BusesServiceImpl(busesServiceStubFactory)
}
