package ru.yandex.travel.bus.service

import com.google.protobuf.Message
import io.grpc.Status
import io.grpc.StatusRuntimeException
import ru.yandex.travel.buses.backend.proto.api.CreateRideOfferRequest
import ru.yandex.travel.buses.backend.proto.api.CreateRideOfferResponse
import ru.yandex.travel.buses.backend.proto.api.GetOfferRequest
import ru.yandex.travel.buses.backend.proto.api.GetOfferResponse
import ru.yandex.travel.buses.backend.proto.worker.TBookRequest
import ru.yandex.travel.buses.backend.proto.worker.TBookResponse
import ru.yandex.travel.buses.backend.proto.worker.TCancelBookingRequest
import ru.yandex.travel.buses.backend.proto.worker.TCancelBookingResponse
import ru.yandex.travel.buses.backend.proto.worker.TConfirmRequest
import ru.yandex.travel.buses.backend.proto.worker.TConfirmResponse
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoRequest
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoResponse
import ru.yandex.travel.buses.backend.proto.worker.TRefundRequest
import ru.yandex.travel.buses.backend.proto.worker.TRefundResponse
import ru.yandex.travel.buses.backend.proto.worker.TResponseHeader
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.commons.proto.EErrorCode
import java.util.concurrent.CompletableFuture

open class BusesServiceImpl(
    private val busesServiceStubFactory: BusesServiceStubFactory,
) : BusesService {

    override fun rideDetails(request: CreateRideOfferRequest): CompletableFuture<CreateRideOfferResponse> =
        FutureUtils.buildCompletableFuture(busesServiceStubFactory.createRoundRobinStub().createRideOffer(request))

    override fun getOffer(request: GetOfferRequest): CompletableFuture<GetOfferResponse> =
        FutureUtils.buildCompletableFuture(busesServiceStubFactory.createRoundRobinStub().getOffer(request))

    override fun book(request: TBookRequest, testContextMsg: Message?): TBookResponse {
        val response: TBookResponse = try {
            busesServiceStubFactory.createRoundRobinStub().book(request).get()
        } catch (exc: StatusRuntimeException) {
            throw translateException(exc)
        }
        rethrowHeaderErrors(response.header)
        return response
    }

    override fun confirm(request: TConfirmRequest): TConfirmResponse {
        val response: TConfirmResponse = try {
            busesServiceStubFactory.createRoundRobinStub().confirm(request).get()
        } catch (exc: StatusRuntimeException) {
            throw translateException(exc)
        }
        rethrowHeaderErrors(response.header)
        return response
    }

    override fun cancelBooking(request: TCancelBookingRequest): TCancelBookingResponse {
        val response: TCancelBookingResponse = try {
            busesServiceStubFactory.createRoundRobinStub().cancelBooking(request).get()
        } catch (exc: StatusRuntimeException) {
            throw translateException(exc)
        }
        rethrowHeaderErrors(response.header)
        return response
    }

    override fun refundInfo(request: TRefundInfoRequest): TRefundInfoResponse {
        val response: TRefundInfoResponse = try {
            busesServiceStubFactory.createRoundRobinStub().refundInfo(request).get()
        } catch (exc: StatusRuntimeException) {
            throw translateException(exc)
        }
        rethrowHeaderErrors(response.header)
        return response
    }

    override fun refund(request: TRefundRequest): TRefundResponse {
        val response: TRefundResponse = try {
            busesServiceStubFactory.createRoundRobinStub().refund(request).get()
        } catch (exc: StatusRuntimeException) {
            throw translateException(exc)
        }
        rethrowHeaderErrors(response.header)
        return response
    }

    private fun translateException(exc: StatusRuntimeException): RuntimeException {
        return if (exc.status.code == Status.Code.UNAVAILABLE) {
            BusesServiceRetryableException(exc)
        } else exc
    }

    companion object {
        @JvmStatic
        fun rethrowHeaderErrors(header: TResponseHeader) {
            val responseCode = header.code
            val errorMessage = header.code.descriptorForType.fullName + ": " + header.error.message
            // todo(maxim-k): TRAVELBACK-1296: separate worker codes
            if (responseCode == EErrorCode.EC_UNAVAILABLE) {
                throw BusesServiceRetryableException(responseCode, errorMessage)
            }
            if (responseCode != EErrorCode.EC_OK) {
                throw BusesServiceException(responseCode, errorMessage)
            }
        }
    }
}
