package ru.yandex.travel.suburban.model;

import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import lombok.extern.jackson.Jacksonized;
import org.javamoney.moneta.Money;

import ru.yandex.travel.suburban.partners.SuburbanCarrier;
import ru.yandex.travel.suburban.partners.SuburbanProvider;
import ru.yandex.travel.suburban.partners.SuburbanTicketBarcodePreset;

@Data
@Builder(toBuilder = true)
@Jacksonized
@JsonIgnoreProperties(ignoreUnknown = true)
public class SuburbanReservation {
    // initial fields
    @NonNull
    private SuburbanProvider provider;

    @NonNull
    private SuburbanCarrier carrier;

    @NonNull
    private Station stationFrom;

    @NonNull
    private Station stationTo;

    // inited, but can be changed
    @NonNull
    private Money price;

    // workflow technical information
    private WorkflowData workflowData;

    // provider-specific data
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private MovistaReservation movistaReservation;

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private ImReservation imReservation;

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private AeroexpressReservation aeroexpressReservation;

    // misc
    private Error error;
    private SuburbanDicts suburbanDicts;

    @JsonIgnore
    public SuburbanProvider getProviderCode() {
        return provider;
    }

    @JsonIgnore
    public String getCarrierCode() {
        return carrier.getValue();
    }

    //ToDo: Сделать базовый класс для всех Reservation, сделать следующие методы в нем виртуальными
    @JsonIgnore
    public String getProviderOrderId() {

        if (provider == SuburbanProvider.AEROEXPRESS) {
            Long orderId = aeroexpressReservation.getOrderId();
            return orderId != null ? orderId.toString() : null;
        }
        else if (provider == SuburbanProvider.MOVISTA ) {
            Integer orderId = movistaReservation.getOrderId();
            return orderId != null ? orderId.toString() : null;
        }
        else if (provider == SuburbanProvider.IM ) {
            Integer orderId = imReservation.getOrderId();
            return orderId != null ? orderId.toString() : null;
        }
        else {
            throw new RuntimeException(String.format("No data is available for %s", provider));
        }
    }

    @JsonIgnore
    public String getCpaProviderOrderId() {
        if (provider == SuburbanProvider.AEROEXPRESS) {
            Long ticketId = aeroexpressReservation.getTicketId();
            return ticketId != null ? ticketId.toString() : null;
        }
        return getProviderOrderId();
    }

    @JsonIgnore
    public String getProviderTicketNumber() {
        if (provider == SuburbanProvider.MOVISTA ) {
            Integer ticketNumber = movistaReservation.getTicketNumber();
            return ticketNumber != null ? ticketNumber.toString() : null;
        } else if (provider == SuburbanProvider.IM) {
            return imReservation.getTicketNumber();
        } else if (provider == SuburbanProvider.AEROEXPRESS) {
            Long ticketNumber = aeroexpressReservation.getTicketId();
            return ticketNumber != null ? ticketNumber.toString() : null;
        } else {
            throw new RuntimeException(String.format("No data is available for %s", provider));
        }
    }

    @JsonIgnore
    public String getProviderTicketBody() {
        if (provider == SuburbanProvider.MOVISTA ) {
            return movistaReservation.getTicketBody();
        } else if (provider == SuburbanProvider.IM) {
            return imReservation.getTicketBody();
        } else if (provider == SuburbanProvider.AEROEXPRESS) {
            return aeroexpressReservation.getTicketBody();
        } else {
            throw new RuntimeException(String.format("No data is available for %s", provider));
        }
    }

    @JsonIgnore
    public SuburbanTicketFlow getSuburbanTicketFlow() {
        if (provider == SuburbanProvider.MOVISTA ) {
            return SuburbanTicketFlow.VALIDATOR;
        } else if (provider == SuburbanProvider.IM) {
            return SuburbanTicketFlow.SIMPLE;
        } else if (provider == SuburbanProvider.AEROEXPRESS) {
            return SuburbanTicketFlow.AEROEXPRESS;
        } else {
            throw new RuntimeException(String.format("No data is available for %s", provider));
        }
    }

    @JsonIgnore
    public SuburbanTicketBarcodePreset getSuburbanTicketBarcodePreset() {
        switch (carrier) {
            case CPPK:
                return SuburbanTicketBarcodePreset.PDF417_CPPK;
            case SZPPK:
            case BASHPPK:
            case SODRUZHESTVO:
                return SuburbanTicketBarcodePreset.PDF417_SZPPK;
            case MTPPK:
                return SuburbanTicketBarcodePreset.AZTEC_MTPPK;
            case AEROEXPRESS:
                return SuburbanTicketBarcodePreset.NO_BARCODE;
        }
        throw new RuntimeException(String.format("No data is available for carrier %s", carrier));
    }

    @JsonIgnore
    public ZonedDateTime getDepartureDateTime() {
        if (provider == SuburbanProvider.MOVISTA ) {
            return movistaReservation.getDate() == null
                    ? null
                    : movistaReservation.getDate().atStartOfDay(ZoneId.of("Europe/Moscow"));
        } else if (provider == SuburbanProvider.IM) {
            return imReservation.getDate().atStartOfDay(ZoneId.of("Europe/Moscow"));
        } else if (provider == SuburbanProvider.AEROEXPRESS) {
            return aeroexpressReservation.getDate().atStartOfDay(ZoneId.of("Europe/Moscow"));
        } else {
            throw new RuntimeException(String.format("No data is available for %s", provider));
        }
    }

    @JsonIgnore
    public WicketDevice getWicket() {
        if (provider == SuburbanProvider.MOVISTA ) {
            return movistaReservation.getWicket();
        } else {
            return null;
        }
    }

    // Запись системных параметров Workflow
    @JsonIgnore
    public void setWorkflowData() {
        workflowData = new WorkflowData();
        workflowData.setConfirmRequestUuid(UUID.randomUUID());
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    @Jacksonized
    public static class Error {
        String message;
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    @Jacksonized
    public static class Station {
        private Integer id;
        private String titleDefault;
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    @Jacksonized
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class WorkflowData {
        UUID confirmRequestUuid;
        Instant confirmStartedAt;
        Instant getTicketBarcodeStartedAt;
    }
}
