package ru.yandex.travel.suburban.partners.movista;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeoutException;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.IAsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.masking.LogAwareBodyGenerator;
import ru.yandex.travel.commons.logging.masking.LogAwareRequestBuilder;
import ru.yandex.travel.suburban.exceptions.SuburbanException;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaParseResponseException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaRequestException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaUnknownException;

import static org.asynchttpclient.util.HttpConstants.Methods;


@Slf4j
public class DefaultMovistaClient implements MovistaClient {
    @NotNull
    private final Config conf;
    private final AsyncHttpClientWrapper asyncHttpClient;
    private final ObjectMapper objectMapper;

    @Valid
    @Data
    @RequiredArgsConstructor
    @AllArgsConstructor
    public static class Config {
        @NotNull
        private String host;

        @NotNull
        private String token;

        @NotNull
        private Duration timeout;

        @NotNull
        private Duration bookTimeout;
    }

    public DefaultMovistaClient(AsyncHttpClientWrapper asyncHttpClient, Config conf) {
        this.conf = conf;
        this.asyncHttpClient = asyncHttpClient;
        this.objectMapper = createObjectMapper();
    }

    @Override
    public MovistaModel.OrderResponse book(MovistaModel.BookRequest request) {
        return sync(sendRequest(
                Methods.POST, Endpoint.BOOK, request, MovistaModel.OrderResponse.class, conf.bookTimeout
        ));
    }

    @Override
    public MovistaModel.OrderResponse confirm(MovistaModel.ConfirmRequest request) {
        return sync(sendRequest(
                Methods.POST, Endpoint.CONFIRM, request, MovistaModel.OrderResponse.class, conf.timeout
        ));
    }

    @Override
    public MovistaModel.OrderResponse orderInfo(MovistaModel.OrderInfoRequest request) {
        return sync(sendRequest(
                Methods.POST, Endpoint.INFO, request, MovistaModel.OrderResponse.class, conf.timeout
        ));
    }

    @Override
    public byte[] getBlankPdf(MovistaModel.BlankPdfRequest request) {
        return sync(sendRequest(
                Methods.POST, Endpoint.BLANK_PDF, request, byte[].class, conf.timeout
        ));
    }

    private <RQ, RS> CompletableFuture<RS> sendRequest(String http_method, Endpoint endpoint, RQ request,
                                                       Class<RS> responseType,
                                                       Duration timeout) {
        var requestBuilder = (LogAwareRequestBuilder) new LogAwareRequestBuilder()
                .setMethod(http_method)
                .setUrl(conf.host + endpoint.getPath())
                .setBody(LogAwareBodyGenerator.of(request, objectMapper))
                .setHeader("Content-Type", "application/json")
                .setHeader("token", conf.token)
                .setReadTimeout(Math.toIntExact(timeout.toMillis()))
                .setRequestTimeout(Math.toIntExact(timeout.toMillis()));

        // TODO: Здесь должно быть endpoint.toString(), чтобы в соломоне записи метились псевдонимами, а не урлами
        return asyncHttpClient.executeRequest(requestBuilder, endpoint.getPath(), null,
                (IAsyncHttpClientWrapper.ResponseParser<RS>)  r -> handleResponse(r, responseType));
    }

    private <T> T handleResponse(Response response, Class<T> resultClass) {
        int status_code = response.getStatusCode();
        if (status_code >= 200 && status_code < 300) {
            return parseResponse(response, resultClass);
        } else if (status_code >= 400 && status_code < 500) {
            MovistaModel.ErrorResponse errorResponse = parseResponse(response, MovistaModel.ErrorResponse.class);
            throw new MovistaRequestException(
                    errorResponse.code,
                    String.format("Request problem: %s: %s", status_code, errorResponse));
        } else if (status_code == 503) {
            var errorResponse = parseResponse(response, MovistaModel.ErrorResponse.class);
            throw new SuburbanRetryableException(
                    String.format("Server problem: %s: %s", status_code, errorResponse));
        } else if (status_code == 502 || status_code == 504) {
            throw new SuburbanRetryableException(String.format("Failed to get response: %s", status_code));
        } else {
            throw new MovistaUnknownException(String.format("Unknown status: %s", status_code));
        }
    }

    private <T> T parseResponse(Response response, Class<T> resultClass) {
        try {
            if (resultClass == byte[].class) {
                return (T) response.getResponseBodyAsBytes();
            }
            return objectMapper.readValue(response.getResponseBody(), resultClass);
        } catch (IOException ex) {
            throw new MovistaParseResponseException(ex);
        }
    }

    public static <T> T sync(CompletionStage<T> future) {
        try {
            return future.toCompletableFuture().join();
        } catch (CompletionException ex) {
            Throwable cause = ex.getCause();
            if (cause instanceof SuburbanException) {
                throw (SuburbanException) cause;
            } else if (cause instanceof TimeoutException) {
                throw new SuburbanRetryableException("call timeout", cause);
            } else if (cause instanceof IOException) {
                throw new SuburbanRetryableException("call io error", cause);
            } else {
                throw new MovistaUnknownException(cause);
            }
        }
    }

    public static ObjectMapper createObjectMapper() {
        return new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .setPropertyNamingStrategy(PropertyNamingStrategy.LOWER_CAMEL_CASE)
                .registerModule(new JavaTimeModule())
                .configure(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY, true)
                .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    }
}
