package ru.yandex.travel.train.model;

import java.time.Instant;
import java.util.Objects;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import org.javamoney.moneta.Money;

import ru.yandex.travel.train.partners.im.model.ImBlankStatus;

public class TrainModelHelpers {
    private TrainModelHelpers() {}

    public static boolean hasTicketsWithBlankStatus(TrainReservation payload, ImBlankStatus blankStatus) {
        return payload.getPassengers().stream().anyMatch(x -> x.getTicket() != null &&
                x.getTicket().getImBlankStatus() == blankStatus);
    }

    public static Instant calculateCanRefundOnServiceTill(TrainReservation payload) {
// Когда в заказе есть билеты с разным статусом ЭР (например, для одного билета ЭР есть, для другого нет),
// нужно возвращать предупреждения для билета, у которого нет ЭР (для этого билета времени на возврат больше).
        return payload.getPassengers().stream().filter(p -> p.getTicket() != null)
                .map(p -> p.getTicket().calculateCanReturnTill(payload))
                .filter(Objects::nonNull).max(Instant::compareTo).orElse(null);
    }

    public static Instant calculateCanChangeElectronicRegistrationTill(TrainReservation payload) {
        var erTill = payload.getPassengers().stream().filter(p -> p.getTicket() != null)
                .map(p -> p.getTicket().calculateCanChangeElectronicRegistrationTill(payload.getDepartureTime(),
                        payload.getReservationRequestData().isElectronicRegistrationEnabled()))
                .filter(Objects::nonNull).max(Instant::compareTo).orElse(null);
        if (erTill == null) {
            erTill = payload.getCanChangeElectronicRegistrationTill();
        }
        return erTill;
    }

    public static boolean checkAllTicketsRefunded(TrainReservation payload) {
        return payload.getPassengers().stream().allMatch(x -> x.getTicket() != null &&
                x.getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED);
    }

    public static Money calculateTotalCostForPassenger(TrainReservation reservation, TrainPassenger passenger) {
        if (reservation.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
            return passenger.calculateTotalCostWithInsurance();
        } else {
            return passenger.getTicket().calculateTotalCost();
        }
    }

    public static CurrencyUnit checkAndGetOrderCurrency(TrainReservation payload) {
        String errorMessage = "Train reservation has different currencies: %s | %s";
        CurrencyUnit currency = null;
        for (TrainPassenger passenger : payload.getPassengers()) {
            if (passenger.getTicket() != null) {
                TrainTicket ticket = passenger.getTicket();
                CurrencyUnit ticketCurrency = ticket.calculateTotalCost().getCurrency();
                if (currency == null) {
                    currency = ticketCurrency;
                }
                Preconditions.checkState(currency.equals(ticketCurrency),
                        String.format(errorMessage, currency, ticketCurrency));
                if (ticket.getPartnerFee() != null) {
                    Preconditions.checkState(currency.equals(ticket.getPartnerFee().getCurrency()),
                            String.format(errorMessage, currency, ticket.getPartnerFee().getCurrency()));
                }
                if (ticket.getPartnerRefundFee() != null) {
                    Preconditions.checkState(currency.equals(ticket.getPartnerRefundFee().getCurrency()),
                            String.format(errorMessage, currency, ticket.getPartnerRefundFee().getCurrency()));
                }
                if (payload.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && passenger.getInsurance() != null && passenger.getInsurance().getAmount() != null) {
                    CurrencyUnit insuranceCurrency = passenger.getInsurance().getAmount().getCurrency();
                    Preconditions.checkState(currency.equals(insuranceCurrency),
                            String.format(errorMessage, currency, insuranceCurrency));
                }
            }
        }
        return currency;
    }
}
