package ru.yandex.travel.train.model;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import ru.yandex.travel.train.partners.im.model.orderinfo.BoardingSystemType;

@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@JsonIgnoreProperties(ignoreUnknown = true)
public class TrainReservation {
    private UUID offerId;
    private String correlationId; // TODO: temporary solution to glue train segments together
    private int segmentIndex;
    private Direction direction;
    private FeeSource feeSource;
    @Deprecated
    private String banditContext;
    private String banditToken;
    private String banditType;
    private Long banditVersion;
    private Integer permille;
    private List<TrainPassenger> passengers;
    private String stationFromCode;
    private String stationFromTimezone;
    private String stationFromRailwayTimezone;
    private int countryFromId;
    private int countryToId;
    private String stationToCode;
    private String stationToTimezone;
    private String stationToRailwayTimezone;
    private Instant departureTime;
    private Instant arrivalTime;
    private String trainNumber;
    private String carNumber;
    private CarType carType;
    private TrainReservationRequestData reservationRequestData;
    private TrainReservationUiData uiData;
    private String reservationNumber;
    private int partnerOrderId;
    @Deprecated(since = "For new orders use passengers[i].ticket.partnerBuyOperationId or getPartnerBuyOperationIds()")
    private Integer partnerBuyOperationId;
    private int partnerItemIndex; // legacy partner index, now this is an internal index for determining master/slave
    private boolean partnerMultiOrder;
    private String partnerDescription;
    private String timeNotice;
    private String specialNotice;
    private String carrier;
    private String companyTitle;
    private boolean isSuburban;
    private boolean onlyFullReturnPossible;
    private UUID isRebookingFor;
    @Deprecated
    private Instant canChangeElectronicRegistrationTill;
    private BoardingSystemType boardingSystemType;

    private InsuranceStatus insuranceStatus;
    private InsuranceStatus targetInsuranceStatus;
    private int checkConfirmationCounter;
    private Instant maxPendingTill;
    private ErrorInfo errorInfo;
    private boolean isMoscowRegion;
    private RoutePolicy routePolicy;
    private boolean isCppk;
    private String provider;
    private boolean mobile;
    private boolean separatePassengersReserving;

    private TrainDicts trainDicts;

    @JsonIgnore
    public boolean isMasterItem() {
        return partnerMultiOrder && partnerItemIndex == 0;
    }

    @JsonIgnore
    public boolean isSlaveItem() {
        return partnerMultiOrder && partnerItemIndex > 0;
    }

    @JsonIgnore
    public Boolean isProviderP2() {
        return "P2".equals(provider);
    }

    @JsonIgnore
    public Boolean backwardWayInRoundTripWithDiscount() {
        return passengers.stream().anyMatch(x -> x.getTicket() != null && x.getTicket().backwardWayInRoundTripWithDiscount());
    }

    @JsonIgnore
    public Boolean isFullRefundPossible() {
        return passengers.stream().filter(p -> p.getCategory() != PassengerCategory.BABY)
                .map(TrainPassenger::isNonRefundableTariff).distinct().toArray().length == 1;
    }

    @JsonIgnore
    public List<Integer> getPartnerBuyOperationIds() {
        List<Integer> operationIds = passengers.stream().filter(p -> p.getTicket() != null)
                .map(p -> p.getTicket().getPartnerBuyOperationId())
                .filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (operationIds.size() == 0 && partnerBuyOperationId != null) {
            // fallback for old orders
            return List.of(partnerBuyOperationId);
        }
        return operationIds;
    }

    @JsonIgnore
    public Map<Integer, Set<Integer>> getBlankIdsByImOrderItemId() {
        Map<Integer, Set<Integer>> blankIdsByImOrderItemId = getPassengers().stream()
                .filter(p -> p.getTicket().getPartnerBuyOperationId() != null)
                .collect(Collectors.groupingBy(p -> p.getTicket().getPartnerBuyOperationId(),
                        Collectors.mapping(p -> p.getTicket().getBlankId(), Collectors.toSet())));
        if (blankIdsByImOrderItemId.size() == 0 && partnerBuyOperationId != null) {
            // fallback for old orders
            blankIdsByImOrderItemId = Map.of(
                    getPartnerBuyOperationId(),
                    getPassengers().stream().map(p -> p.getTicket().getBlankId()).collect(Collectors.toSet())
            );
        }
        return blankIdsByImOrderItemId;
    }
}
