package ru.yandex.travel.train.model;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.TemporalUnit;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.BoardingSystemType;

@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class TrainTicket {
    private int blankId;
    private Integer partnerBuyOperationId;
    private List<TrainPlace> places;
    private TrainTicketRefundStatus refundStatus;
    private ImBlankStatus imBlankStatus; // rzhdStatus
    private boolean pendingElectronicRegistration;

    @Builder.Default
    private Money tariffAmount = Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB); // Transportation cost
    private Money tariffVatAmount;
    private Double tariffVatRate;
    private Integer tariffFiscalItemInternalId;

    @Builder.Default
    private Money serviceAmount = Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB); // Additional carrier services
    // (bedding)
    private Money serviceVatAmount;
    private Double serviceVatRate;
    private Integer serviceFiscalItemInternalId;

    private boolean isBanditFeeApplied;
    @Builder.Default
    private Money feeAmount = Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB); // Yandex fee
    private Integer feeFiscalItemInternalId;

    private String rawTariffType;
    private String rawTariffName;
    private String bookedTariffCode;
    private String carrierInn;

    private Money partnerFee;
    private Money partnerRefundFee;

    private Instant canChangeElectronicRegistrationTill;
    private Instant canReturnTill;

    public Instant calculateCanChangeElectronicRegistrationTill(Instant departure, boolean enabled) {
        if (!enabled) {
            return null;
        }
        if (imBlankStatus == ImBlankStatus.NO_REMOTE_CHECK_IN) {
            return departure;
        }
        return canChangeElectronicRegistrationTill;
    }

    @JsonIgnore
    public Instant calculateCanReturnTill(TrainReservation trainReservation) {
        if (canReturnTill == null) {
            if (trainReservation.isProviderP2()) {
                return trainReservation.getDepartureTime() != null ?
                        trainReservation.getDepartureTime().minus(Duration.ofHours(1)) : null;
            }
            if (trainReservation.getBoardingSystemType() == BoardingSystemType.PASSENGER_BOARDING_CONTROL) {
                return trainReservation.getDepartureTime();
            }
        }
        return canReturnTill;
    }

    @JsonIgnore
    public Money calculateTotalCost() {
        return getTariffAmount().add(getServiceAmount()).add(getFeeAmount());
    }

    @JsonIgnore
    public Money calculateRefundFeeAmount() {
        return getFeeAmount().subtract(getPartnerFee()).subtract(getPartnerRefundFee());
    }

    @JsonIgnore
    public Money calculateRefundFeeAmountOrNull() {
        if (getPartnerFee() != null && getPartnerRefundFee() != null) {
            return calculateRefundFeeAmount();
        } else {
            return null;
        }
    }

    @JsonIgnore
    public boolean backwardWayInRoundTripWithDiscount() {
        return rawTariffType != null && rawTariffType.equals("BackwardWayInRoundTripWithDiscount");
    }
}
