package ru.yandex.travel.train.partners.im;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Realm;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationRequest;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.ImErrorResponse;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCancelRequest;
import ru.yandex.travel.train.partners.im.model.ReservationConfirmRequest;
import ru.yandex.travel.train.partners.im.model.ReservationConfirmResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountRequest;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.UpdateBlanksRequest;
import ru.yandex.travel.train.partners.im.model.UpdateBlanksResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoRequest;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListRequest;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListResponse;

import static ru.yandex.travel.train.partners.im.ImClientHelpers.sync;

@Slf4j
public class DefaultImClient implements ImClient {
    private static final String DEFAULT_PAYMENT_FORM = "Card";

    private final AsyncHttpClientWrapper asyncHttpClient;
    private final ObjectMapper objectMapper;
    private final String pos;
    private final Duration timeout;
    private final String baseUrl;
    private final String login;
    private final String password;

    public DefaultImClient(AsyncHttpClientWrapper asyncHttpClient, String pos, Duration timeout, String baseUrl,
                           String login, String password) {
        this.asyncHttpClient = asyncHttpClient;
        this.pos = pos;
        this.baseUrl = baseUrl;
        this.login = login;
        this.password = password;
        this.timeout = timeout;
        this.objectMapper = createImObjectMapper();
    }

    public static ObjectMapper createImObjectMapper() {
        return new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .setPropertyNamingStrategy(PropertyNamingStrategy.UPPER_CAMEL_CASE)
                .registerModule(new JavaTimeModule())
                .configure(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY, true)
                .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
    }

    @Override
    public AutoReturnResponse autoReturn(AutoReturnRequest request) {
        return sync(sendRequest(
                "POST", Method.AUTO_RETURN, request, AutoReturnResponse.class, this.timeout
        ));
    }

    @Override
    public ElectronicRegistrationResponse changeElectronicRegistration(ElectronicRegistrationRequest request) {
        return sync(sendRequest(
                "POST", Method.ELECTRONIC_REGISTRATION, request, ElectronicRegistrationResponse.class,
                this.timeout
        ));
    }

    @Override
    public InsuranceCheckoutResponse insuranceCheckout(InsuranceCheckoutRequest request) {
        return sync(sendRequest(
                "POST", Method.INSURANCE_CHECKOUT, request, InsuranceCheckoutResponse.class, this.timeout
        ));
    }

    @Override
    public InsurancePricingResponse insurancePricing(InsurancePricingRequest request) {
        return sync(sendRequest(
                "POST", Method.INSURANCE_PRICING, request, InsurancePricingResponse.class, this.timeout
        ));
    }

    @Override
    public InsuranceReturnResponse insuranceReturn(InsuranceReturnRequest request) {
        return sync(sendRequest(
                "POST", Method.INSURANCE_RETURN, request, InsuranceReturnResponse.class, this.timeout
        ));
    }

    @Override
    public CompletableFuture<byte[]> orderReservationBlankAsync(OrderReservationBlankRequest request,
                                                                Duration timeout) {
        return sendRequest(
                "POST", Method.ORDER_RESERVATION_BLANK, request, byte[].class, timeout != null ? timeout : this.timeout
        ).thenApply(response -> {
            if (response == null || response.length == 0) {
                throw new ImClientParseException("Unexpected empty blank body");
            }
            return response;
        });
    }

    @Override
    public CompletableFuture<OrderReservationTicketBarcodeResponse> orderReservationTicketBarcodeAsync(OrderReservationTicketBarcodeRequest request,
                                                                                                       Duration timeout) {
        return sendRequest(
                "POST", Method.ORDER_RESERVATION_TICKET_BARCODE, request,
                OrderReservationTicketBarcodeResponse.class, timeout != null ? timeout : this.timeout
        );
    }

    @Override
    public ReservationCreateResponse reservationCreate(ReservationCreateRequest request, Object data) {
        return sync(sendRequest(
                "POST", Method.RESERVE, request, ReservationCreateResponse.class, this.timeout
        ));
    }

    @Override
    public void reservationCancel(int orderId) {
        var request = new ReservationCancelRequest();
        request.setOrderId(orderId);
        sync(sendRequest(
                "POST", Method.RESERVATION_CANCEL, request, null, this.timeout
        ));
    }

    @Override
    public ReservationConfirmResponse reservationConfirm(int orderId) {
        var request = new ReservationConfirmRequest();
        request.setOrderId(orderId);
        request.setProviderPaymentForm(DEFAULT_PAYMENT_FORM);
        return sync(sendRequest(
                "POST", Method.RESERVATION_CONFIRM, request, ReservationConfirmResponse.class, this.timeout
        ));
    }

    @Override
    public ReturnAmountResponse getReturnAmount(ReturnAmountRequest request) {
        return sync(sendRequest(
                "POST", Method.RETURN_AMOUNT, request, ReturnAmountResponse.class, this.timeout
        ));
    }

    @Override
    public CompletableFuture<OrderInfoResponse> orderInfoAsync(int orderId, Duration timeout) {
        var request = new OrderInfoRequest();
        request.setOrderId(orderId);
        return sendRequest(
                "POST", Method.ORDER_INFO, request, OrderInfoResponse.class, timeout != null ? timeout : this.timeout
        );
    }

    @Override
    public OrderListResponse orderList(OrderListRequest request) {
        return sync(sendRequest("POST", Method.ORDER_LIST, request, OrderListResponse.class, this.timeout));
    }

    @Override
    public UpdateBlanksResponse updateBlanks(int orderItemId, Duration timeout) {
        var request = new UpdateBlanksRequest();
        request.setOrderItemId(orderItemId);
        return sync(sendRequest("POST", Method.UPDATE_BLANKS, request, UpdateBlanksResponse.class,
                timeout != null ? timeout : this.timeout));
    }

    private <RQ, RS> CompletableFuture<RS> sendRequest(String method, Method purpose, RQ body, Class<RS> responseType,
                                                       Duration timeout) {
        String bodyString = serializeRequest(body);
        RequestBuilder requestBuilder = createBaseRequestBuilder(method, timeout)
                .setUrl(baseUrl + purpose.getPath())
                .setBody(body != null ? bodyString : null);
        return asyncHttpClient.executeRequest(requestBuilder, purpose.toString())
                .thenApply(r -> parseResponse(r, responseType));
    }

    private <T> T parseResponse(Response response, Class<T> resultClass) {
        if (response.getStatusCode() >= 200 && response.getStatusCode() < 300) {
            try {
                if (resultClass == byte[].class) {
                    return (T) response.getResponseBodyAsBytes();
                }
                if (resultClass == null) {
                    return null;
                }
                return objectMapper.readValue(response.getResponseBody(), resultClass);
            } catch (IOException e) {
                log.error("Unable to parse response", e);
                throw new ImClientParseException("Unable to parse response", e);
            }
        } else if (response.getStatusCode() >= 502 && response.getStatusCode() <= 504) {
            throw new ImClientIOException(String.format("Error get response from IM. httpCode = %s",
                    response.getStatusCode()));
        } else if (response.getStatusCode() >= 500) {
            try {
                ImErrorResponse imError = objectMapper.readValue(response.getResponseBody(), ImErrorResponse.class);
                throw ImErrorsHelper.getExceptionFromImError(imError);
            } catch (IOException e) {
                log.error("Unable to parse error response", e);
                throw new ImClientParseException("Unable to parse error response", e);
            }
        } else {
            throw new ImClientParseException(String.format("Do not know how to handle status %s from im",
                    response.getStatusCode()));
        }
    }

    private <T> String serializeRequest(T request) {
        try {
            return objectMapper.writeValueAsString(request);
        } catch (JsonProcessingException e) {
            log.error("Error serializing request", e);
            throw new RuntimeException(e);
        }
    }

    private RequestBuilder createBaseRequestBuilder(String method, Duration timeout) {
        Realm realm = new Realm.Builder(login, password)
                .setScheme(Realm.AuthScheme.BASIC)
                .setUsePreemptiveAuth(true)
                .build();

        RequestBuilder builder = new RequestBuilder()
                .setHeader("POS", pos)
                .setHeader("Content-Type", "application/json")
                .setRealm(realm)
                .setReadTimeout(Math.toIntExact(timeout.toMillis()))
                .setRequestTimeout(Math.toIntExact(timeout.toMillis()))
                .setMethod(method);
        return builder;
    }

    public enum Method {
        AUTO_RETURN("Order/V1/Reservation/AutoReturn"),
        ELECTRONIC_REGISTRATION("Railway/V1/Reservation/ElectronicRegistration"),
        INSURANCE_CHECKOUT("Insurance/V1/Travel/Checkout"),
        INSURANCE_PRICING("Insurance/V1/Travel/Pricing"),
        INSURANCE_RETURN("Insurance/V1/Travel/Return"),
        ORDER_INFO("Order/V1/Info/OrderInfo"),
        ORDER_LIST("Order/V1/Info/OrderList"),
        ORDER_RESERVATION_BLANK("Order/V1/Reservation/Blank"),
        ORDER_RESERVATION_TICKET_BARCODE("Railway/V1/Reservation/TicketBarcode"),
        RESERVATION_CANCEL("Order/V1/Reservation/Cancel"),
        RESERVATION_CONFIRM("Order/V1/Reservation/Confirm"),
        RESERVE("Order/V1/Reservation/Create"),
        RETURN_AMOUNT("Order/V1/Reservation/ReturnAmount"),
        UPDATE_BLANKS("Railway/V1/Reservation/UpdateBlanks");

        private final String path;

        Method(String path) {
            this.path = path;
        }

        public String getPath() {
            return path;
        }
    }
}
