package ru.yandex.travel.train.partners.im;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.databind.JsonNode;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import ru.yandex.travel.train.partners.im.model.ImErrorResponse;

import javax.validation.constraints.NotNull;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ImErrorsHelper {

    @RequiredArgsConstructor
    public static class ErrorCaseChecker {

        @NotNull
        public final List<Pair<Integer, Pattern>> cases;

        public String Check(ImErrorResponse imError) {
            for (Pair<Integer, Pattern> codePattern : cases) {
                if (imError.getCode() != codePattern.getLeft()) {
                    continue;
                }
                var pattern = codePattern.getRight();
                if (pattern == null)
                    return "";
                if (imError.getProviderError() != null) {
                    Matcher matcher = pattern.matcher(imError.getProviderError());
                    if (matcher.find()) {
                        if (matcher.groupCount() > 0) {
                            return matcher.group(0);
                        }
                        return "";
                    }
                }
                for (String messageParam : jsonArrayToStringList(imError.getMessageParams())) {
                    Matcher matcher = pattern.matcher(messageParam);
                    if (matcher.find()) {
                        if (matcher.groupCount() > 0) {
                            return matcher.group(0);
                        }
                        return "";
                    }
                }
            }
            return null;
        }
    }

    // Transport errors
    public static final ErrorCaseChecker PARTNER_ERROR = new ErrorCaseChecker(List.of(
            Pair.of(1, null)
    ));
    public static final ErrorCaseChecker TRY_LATER = new ErrorCaseChecker(List.of(
            Pair.of(2, null),
            Pair.of(3, Pattern.compile("попробуйте повторить позже", Pattern.CASE_INSENSITIVE)),
            Pair.of(94, null)
    ));

    // User data errors
    public static final ErrorCaseChecker NO_PLACES = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("Код ошибки провайдера\\: 346\\.")),
            Pair.of(3, Pattern.compile("ВАГОН ЖЕНСКИЙ")),
            Pair.of(3, Pattern.compile("ВАГОН МУЖСКОЙ")),
            Pair.of(310, null),
            Pair.of(311, null),
            Pair.of(342, null),
            Pair.of(345, null),
            Pair.of(346, null)
    ));
    public static final ErrorCaseChecker INVALID_BONUS_CARD = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("US330:")),
            Pair.of(3, Pattern.compile("US331:")),
            Pair.of(1461, Pattern.compile("US333:")),
            Pair.of(364, null),
            Pair.of(365, null),
            Pair.of(367, null)
    ));
    public static final ErrorCaseChecker NAME_REQUIRED_LATIN_LETTERS = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("В ФИО ПАССАЖИРА BCE БУКВЫ Д\\.Б\\. ЛАТИНСКИМИ"))
    ));
    public static final ErrorCaseChecker INVALID_DOCUMENT_NUMBER = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("US194:")),
            Pair.of(42, Pattern.compile("request.Customers\\[(\\d+)\\].DocumentNumber"))
    ));
    public static final ErrorCaseChecker CITIZENSHIP_NOT_MATCH_DOCUMENT_TYPE = new ErrorCaseChecker(List.of(
            Pair.of(73, Pattern.compile("request.Customers\\[(\\d+)\\].DocumentType"))
    ));
    public static final ErrorCaseChecker INVALID_PASSENGER_PHONE = new ErrorCaseChecker(List.of(
            Pair.of(1385, null)
    ));
    public static final ErrorCaseChecker INVALID_PASSENGER_EMAIL = new ErrorCaseChecker(List.of(
            Pair.of(1386, null)
    ));
    public static final ErrorCaseChecker INVALID_PASSENGER_DATA = new ErrorCaseChecker(List.of(
            Pair.of(1461, null)
    ));

    // Order errors
    public static final ErrorCaseChecker TOO_LATE_FOR_ORDER = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("Электронная продажа билетов приостановлена за ([\\d\\:]+) до отправления"))
    ));
    public static final ErrorCaseChecker TOO_LATE = new ErrorCaseChecker(List.of(
            Pair.of(1312, null)
    ));
    public static final ErrorCaseChecker TARIFF_ERROR = new ErrorCaseChecker(List.of(
            Pair.of(3, Pattern.compile("НЕ ВЫПОЛНЕНЫ УСЛОВИЯ СПЕЦТАРИФА")),
            Pair.of(328, null),
            Pair.of(349, null),
            Pair.of(1350, null)
    ));

    public static ImClientException getExceptionFromImError(ImErrorResponse imError) {

        if (PARTNER_ERROR.Check(imError) != null) {
            return new ImClientIOException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }
        if (TRY_LATER.Check(imError) != null) {
            return new ImClientRetryableException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }

        if (NO_PLACES.Check(imError) != null) {
            return new ImClientNoPlacesException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }
        if (INVALID_BONUS_CARD.Check(imError) != null) {
            return new ImClientBonusCardException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }
        var passengerIndex = NAME_REQUIRED_LATIN_LETTERS.Check(imError);
        if (passengerIndex != null) {
            return new ImClientNameRequiredLatinLettersException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }
        passengerIndex = INVALID_DOCUMENT_NUMBER.Check(imError);
        if (passengerIndex != null) {
            return new ImClientInvalidDocumentNumberException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }
        passengerIndex = CITIZENSHIP_NOT_MATCH_DOCUMENT_TYPE.Check(imError);
        if (passengerIndex != null) {
            return new ImClientCitizenshipNotMatchDocumentTypeException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }
        passengerIndex = INVALID_PASSENGER_PHONE.Check(imError);
        if (passengerIndex != null) {
            return new ImClientInvalidPassengerPhoneException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }
        passengerIndex = INVALID_PASSENGER_EMAIL.Check(imError);
        if (passengerIndex != null) {
            return new ImClientInvalidPassengerEmailException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }
        passengerIndex = INVALID_PASSENGER_DATA.Check(imError);
        if (passengerIndex != null) {
            return new ImClientInvalidPassengerDataException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), Integer.getInteger(passengerIndex, -1));
        }

        var salesClosesBeforeDeparture = TOO_LATE_FOR_ORDER.Check(imError);
        if (salesClosesBeforeDeparture != null) {
            return new ImClientTooLateForOrderException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()), salesClosesBeforeDeparture);
        }
        if (TOO_LATE.Check(imError) != null) {
            return new ImClientTooLateException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }
        if (TARIFF_ERROR.Check(imError) != null) {
            return new ImClientTariffErrorException(imError.getCode(), imError.getMessage(),
                    jsonArrayToStringList(imError.getMessageParams()));
        }

        return new ImClientUnknownPartnerErrorException(imError.getCode(), imError.getMessage(),
                jsonArrayToStringList(imError.getMessageParams()));
    }

    private static List<String> jsonArrayToStringList(JsonNode node) {
        List<String> result = new ArrayList<>();
        if (node == null) {
            return result;
        }
        if (node.isArray()) {
            for (JsonNode child : node) {
                result.add(jsonNodeToString(child));
            }
        } else {
            result.add(jsonNodeToString(node));
        }
        return result;
    }

    private static String jsonNodeToString(JsonNode node) {
        if (node.isTextual()) {
            return node.textValue();
        } else {
            return node.toString();
        }
    }
}
