# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import argparse
import json
import os
import tempfile
import time
import yatest.common

from library.python.testing.recipe import declare_recipe
from travel.library.recipe_utils.utils import (
    check_call, get_free_port, log, run_process, Timer, set_environ, untar
)


POSTGRES_TGZ = 'postgres.tgz'
SHUTDOWN_TIMEOUT = 10
FILENAME = 'postgres_recipe'
DBNAME = 'rasp_postgres_test'
USERNAME = 'rasp'
PG_PROCESS_NAME = 'postgres'


def postgres_is_alive(port, cmd_path, username, env=None):
    cmd = [cmd_path, '--host', 'localhost', '--port', str(port), '--username', username]
    try:
        check_call(cmd, '{}.pg_isready'.format(PG_PROCESS_NAME), env or {})
    except Exception:
        return False
    return True


def wait_for_postgres(process_name, is_alive, attempts=60):
    for attempt in range(attempts):
        log(process_name, 'attempt', attempt, '...')
        result = is_alive()
        if result:
            return
        log('Can\'t connect to {process_name}.'.format(
            process_name=process_name,
        ))
        time.sleep(1)
    raise Exception('Failed to start {}'.format(process_name))


def start(argv):
    log('Setting up PostgreSql')

    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--archive',
        metavar='<file>',
        type=str,
        help='postgres.tgz path',
        default=POSTGRES_TGZ
    )
    args, argv = parser.parse_known_args(argv)
    postgres_dir = tempfile.mkdtemp(prefix='postgres_')
    working_dir = os.path.join(postgres_dir, 'wd')
    os.mkdir(working_dir)

    untar(args.archive, path=postgres_dir)

    bin_dir = os.path.join(postgres_dir, 'postgres', 'bin')
    lib_dir = os.path.join(postgres_dir, 'postgres', 'lib')

    env = os.environ.copy()
    if 'LD_LIBRARY_PATH' in env:
        env['LD_LIBRARY_PATH'] += ':' + lib_dir
    else:
        env['LD_LIBRARY_PATH'] = lib_dir

    env['LANG'] = 'en_US.UTF-8'
    env['LC_MESSAGES'] = 'en_US.UTF-8'

    initdb_path = os.path.join(bin_dir, 'initdb')
    pg_isready_path = os.path.join(bin_dir, 'pg_isready')
    createdb_path = os.path.join(bin_dir, 'createdb')
    psql_path = os.path.join(bin_dir, 'psql')

    postgres_port = get_free_port(desired_port=6432)

    with Timer('Initializing PostgreSql cluster...:'):
        cmd = [initdb_path, '-D', working_dir, '-U', USERNAME]
        check_call(cmd, PG_PROCESS_NAME, env)

    sockets_dir = tempfile.mkdtemp(prefix='sockets_')

    with Timer('Run PostgreSql server...:'):
        postgres_server_path = os.path.join(bin_dir, 'pg_ctl')
        cmd = [
            postgres_server_path,
            '--pgdata', working_dir,
            '--log', yatest.common.output_path('{}_server.log'.format(PG_PROCESS_NAME)),
            '-o', '-p{}'.format(postgres_port),
            '-o', '-k{}'.format(sockets_dir),
            'start'
        ]

        server = run_process(cmd, '{}.pg_ctl'.format(PG_PROCESS_NAME), env)

        with open('{}.pid'.format(FILENAME), 'w') as f:
            f.write(str(server.pid))
        with open('{}.port'.format(FILENAME), 'w') as f:
            f.write(str(postgres_port))

        wait_for_postgres(
            PG_PROCESS_NAME,
            lambda: postgres_is_alive(postgres_port, pg_isready_path, USERNAME, env),
            attempts=60
        )

    with Timer('Create PostgreSql test base...:'):
        cmd = [createdb_path, '--host', 'localhost', '--port', str(postgres_port), '--username', USERNAME, DBNAME]
        check_call(cmd, '{}.createdb'.format(PG_PROCESS_NAME), env)

    set_environ('TRAVEL_POSTGRES_RECIPE_BIN_DIR', bin_dir)
    set_environ('TRAVEL_POSTGRES_RECIPE_SOCKETS_DIR', sockets_dir)
    set_environ('TRAVEL_POSTGRES_RECIPE_PID', str(server.pid))
    set_environ('TRAVEL_POSTGRES_RECIPE_PORT', str(postgres_port))
    set_environ('TRAVEL_POSTGRES_RECIPE_DBNAME', DBNAME)
    set_environ('TRAVEL_POSTGRES_RECIPE_USER', USERNAME)

    with open('pg_recipe.json', 'w') as pg_recipe:
        pg_recipe.write(json.dumps({
            "host": sockets_dir,
            "port": postgres_port,
            "dbname": DBNAME,
            "user": USERNAME,
            "psql": psql_path
        }))

    log('Done setting up PostgreSql.')


def stop(argv):
    postgres_port = int(os.getenv('TRAVEL_POSTGRES_RECIPE_PORT'))
    bin_dir = os.getenv('TRAVEL_POSTGRES_RECIPE_BIN_DIR')

    pg_isready_path = os.path.join(bin_dir, 'pg_isready')
    if not postgres_is_alive(postgres_port, pg_isready_path, USERNAME):
        log('PostgreSql server is already stopped')
        return

    with Timer('Tearing down PostgreSql...:'):
        postgres_server_path = os.path.join(bin_dir, 'pg_ctl')

        cmd = [postgres_server_path, 'stop']
        try:
            check_call(cmd, PG_PROCESS_NAME, {})
        except Exception:
            pass

    try:
        wait_for_postgres(
            PG_PROCESS_NAME,
            lambda: not postgres_is_alive(postgres_port, pg_isready_path, USERNAME),
            attempts=SHUTDOWN_TIMEOUT
        )
    except Exception:
        log('postgres failed to shutdown in {} seconds'.format(SHUTDOWN_TIMEOUT))


if __name__ == "__main__":
    declare_recipe(start, stop)
