# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from travel.library.python.base_http_client import BaseHttpClient


log = logging.getLogger(__name__)


class BarisClient(BaseHttpClient):
    HTTP_CLIENT_NAME = 'BARIS'

    def _call_and_parse(self, url_path, params=None, timeout=None):
        response = self.get(
            'api/v1/{}'.format(url_path),
            params=params or {},
            timeout=timeout
        )
        return response.json()

    def flight_board(self, station_id, direction, after=None, before=None, terminal=None, limit=None):
        params = {'direction': direction}

        # Откидываем из времен таймзону, так как БАРиС ждет даты без таймзоны
        if before:
            params['before'] = before.replace(tzinfo=None).isoformat()
        if after:
            params['after'] = after.replace(tzinfo=None).isoformat()
        if terminal:
            params['terminal'] = terminal
        if limit:
            params['limit'] = limit

        return self._call_and_parse(
            'flight-board/{}/'.format(station_id),
            params=params
        )

    def flight_board_schedule(self, station_id, direction):
        return self._call_and_parse(
            'flight-board/{}/schedule/'.format(station_id),
            params={'direction': direction},
            timeout=10
        )

    def _make_p2p_params(
        self, station_from_ids, station_to_ids, national_version, show_banned, before, after, flight_numbers=None,
    ):
        params = {
            'from': list(station_from_ids),
            'to': list(station_to_ids),
        }
        if national_version:
            params['national_version'] = national_version
        if show_banned:
            params['show_banned'] = show_banned
        if flight_numbers:
            params['flight'] = list(flight_numbers)

        if before:
            params['before'] = before.isoformat()
        if after:
            params['after'] = after.isoformat()

        return params

    def flight_p2p(
        self, station_from_ids, station_to_ids, national_version=None, show_banned=None, before=None, after=None
    ):
        params = self._make_p2p_params(station_from_ids, station_to_ids, national_version, show_banned, before, after)
        return self._call_and_parse(
            'flight-p2p',
            params=params,
        )

    def flight_p2p_with_numbers(
        self, station_from_ids, station_to_ids, national_version=None, show_banned=None, flight_numbers=None,
        before=None, after=None
    ):
        params = self._make_p2p_params(
            station_from_ids, station_to_ids, national_version, show_banned, before, after, flight_numbers
        )
        return self._call_and_parse(
            'flight-p2p',
            params=params,
            timeout=6
        )

    def flight_p2p_schedule(self, station_from_ids, station_to_ids, national_version=None, show_banned=None):
        params = {
            'from': list(station_from_ids),
            'to': list(station_to_ids),
        }
        if national_version:
            params['national_version'] = national_version
        if show_banned:
            params['show_banned'] = show_banned

        return self._call_and_parse(
            'flight-p2p-schedule',
            params=params,
        )

    def flight_schedule(self, flight_number):
        return self._call_and_parse(
            'flight-schedule/{}'.format(flight_number.replace('-', '/')),
        )

    def delayed_flights(self, station_from_ids):
        return self._call_and_parse(
            'delayed-flights',
            params={'station': list(station_from_ids)}
        )

    def flight_p2p_summary(self):
        return self._call_and_parse(
            'flight-p2p-summary',
            timeout=60
        )
