# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import datetime
import logging

from xml.etree import ElementTree

from travel.library.python.base_http_client import BaseHttpClient, RetryConfig, HTTPBasicAuthHeaderCreator


log = logging.getLogger(__name__)


CHELYABINSK_BUS_API_HOST = 'https://production.ipv4-proxy.internal.rasp.yandex.net'
CHELYABINSK_BUS_API_URL = 'сhelyabinsk_bus/newasu/ws/YandexSchedule.1cws'


class ChelyabinskBusClient(BaseHttpClient):
    HTTP_CLIENT_NAME = 'ChelyabinskBus'
    RETRY_CONFIG = RetryConfig(total=5, backoff_factor=5, method_whitelist=['POST'])
    DISABLE_CIRCUIT_BREAKER_CONFIG = True
    TIMEOUT = 120

    def __init__(self, host=CHELYABINSK_BUS_API_HOST, login=None, password=None, **kwargs):
        headers = {'Content-Type': 'application/xml'}

        super(ChelyabinskBusClient, self).__init__(
            host=host,
            headers=headers,
            custom_headers_creators=[HTTPBasicAuthHeaderCreator(login, password)],
            **kwargs
        )

    def call_and_parse(self, data):
        # пример вызова через curl - https://st.yandex-team.ru/RASPFRONT-10048#61a745b295687b3d38b42b2f

        response = self.post(
            url_path=CHELYABINSK_BUS_API_URL,
            data=data
        )

        return response.text

    @staticmethod
    def get_text(node, tag):
        for child in node:
            if child.tag == '{http://www.YaTickets.ru}' + tag:
                return child.text

    def get_buses(self):
        query = '''<?xml version='1.0' encoding='utf-8'?>
        <soap-env:Envelope xmlns:soap-env="http://schemas.xmlsoap.org/soap/envelope/">
        <soap-env:Body>
        <ns0:GetTheBus xmlns:ns0="http://www.YaTickets.ru"/>
        </soap-env:Body>
        </soap-env:Envelope>'''

        response = self.call_and_parse(query)
        xnl = ElementTree.fromstring(response.encode('utf-8'))

        return [{'id': self.get_text(n, 'Id'), 'name': self.get_text(n, 'Name')} for n in xnl[0][0][0]]

    def get_threads(self, bus_id, date):
        query = '''<soap-env:Envelope xmlns:soap-env="http://schemas.xmlsoap.org/soap/envelope/">
        <soap-env:Body>
        <ns0:GetFlightsToBusStation xmlns:ns0="http://www.YaTickets.ru">
        <ns0:BusStationId>{}</ns0:BusStationId>
        <ns0:DepartureData>{}</ns0:DepartureData>
        <ns0:Active>true</ns0:Active>
        <ns0:AdditionalFlight>false</ns0:AdditionalFlight>
        </ns0:GetFlightsToBusStation>
        </soap-env:Body>
        </soap-env:Envelope>'''.format(bus_id, date)

        response = self.call_and_parse(query)
        xml = ElementTree.fromstring(response.encode('utf-8'))

        return [self._create_thread(n) for n in xml[0][0][0]]

    def _create_thread(self, node):
        thread = {
            'Flight': self.get_text(node, 'Flight'),
            'RoutingNumber': self.get_text(node, 'RoutingNumber') or '',
            'ModelBus': self.get_text(node, 'ModelBus'),
            'AutotransportCompany': self.get_text(node, 'AutotransportCompany'),
            'TrackList': []
        }

        for point in [c for c in node if c.tag == '{http://www.YaTickets.ru}TrackList']:
            stop = {
                'Point': self.get_text(point, 'Point'),
                'OffsetDays': self.get_text(point, 'OffsetDays'),
                'DistanceFromTheFirstStop': self.get_text(point, 'DistanceFromTheFirstStop'),
                'ArrivalTime': self.get_text(point, 'ArrivalTime'),
                'DepartureTime': self.get_text(point, 'DepartureTime'),
                'TicketPrice': self.get_text(point, 'TicketPrice')
            }

            thread['TrackList'].append(stop)

        return thread

    def get_schedule(self, bus_id, start_date, days_count):
        schedule = []
        for delta in range(days_count):
            date = start_date + datetime.timedelta(days=delta + 1)
            date_str = date.strftime('%Y-%m-%d')

            schedule.append((date_str, self.get_threads(bus_id, date_str)))

        return schedule
