# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from marshmallow import fields, Schema


class IntergerEmptyField(fields.Integer):
    """
    fields.Integer with parsing of empty string
    """

    def _deserialize(self, value, attr, obj):
        if value == '':
            return None
        return super(IntergerEmptyField, self)._deserialize(value, attr, obj)


class DzvRaspItemSchema(Schema):
    # Номер поезда
    train_number = fields.String(load_from='TrainNumber')
    # Станция отправления
    start_station = fields.String(load_from='StartStation')
    # Станция прибытия
    end_station = fields.String(load_from='EndStation')
    # Номер пути
    track_number = fields.String(load_from='EvTrackNumber')
    # Дата и время события (отправки или отправления)
    event_dt = fields.DateTime(load_from='EvSndTime')
    # Тип поезда:
    # пусто - не определен,
    # 0 - пригородный,
    # 1 - скорый,
    # 2 - скоростной,
    # 3 - фирменный,
    # 4 - пассажирский,
    # 5 - рэкс (пригородный экспресс), 5 - ласточка (пригородный экспресс))
    train_type = IntergerEmptyField(load_from='TrainType')


class DzvRaspSchema(Schema):
    arrival = fields.Nested(DzvRaspItemSchema, many=True)
    departure = fields.Nested(DzvRaspItemSchema, many=True)


class DzvRaspResponseSchema(Schema):
    response = fields.Nested(DzvRaspSchema, allow_none=True)
    error = IntergerEmptyField()
    error_description = fields.String(required=False, default=None)


class DzvResponseError(Exception):
    pass


def parse_platforms(platform_response_content):
    """Десериализация ответа ДЖВ"""

    platforms_data, errors = DzvRaspResponseSchema().loads(platform_response_content)

    if errors:
        raise DzvResponseError('Cannot parse API response: {}'.format(errors))
    if platforms_data['error']:
        raise DzvResponseError(
            'API response with errors: {} - {}'.format(platforms_data['error'], platforms_data['error_description'])
        )
    return platforms_data['response']
