# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import datetime, date

import requests

from travel.library.python.tracing.instrumentation import traced_function
from travel.library.python.base_http_client import (
    BaseHttpClient, CircuitBreakerConfig, HTTPBasicAuthHeaderCreator, RetryConfig
)


log = logging.getLogger(__name__)


class ImClient(BaseHttpClient):
    """
    Клиент для работы с АПИ ИМ (Инновационная мобильность) - партнера для продаж билетов на ЖД
    https://st.yandex-team.ru/RASPFRONT-9742
    """
    HTTP_CLIENT_NAME = 'IM'
    RETRY_CONFIG = RetryConfig(total=3)
    CIRCUIT_BREAKER_CONFIG = CircuitBreakerConfig(fail_max=3, reset_timeout=20)

    def __init__(self, host, username, password, pos, **kwargs):
        self.username = username
        self.password = password
        self.pos = pos

        headers = kwargs.pop('headers', {})
        headers.update({'POS': self.pos})
        super(ImClient, self).__init__(
            host=host,
            headers=headers,
            custom_headers_creators=[HTTPBasicAuthHeaderCreator(self.username, self.password)],
            **kwargs
        )

    @classmethod
    def dt_to_im_date(cls, dt):
        # type: (datetime) -> unicode
        return dt.strftime('%Y-%m-%d')

    @classmethod
    def date_from_im_date(cls, dtstr):
        # type: (unicode) -> date
        return datetime.strptime(dtstr, '%Y-%m-%d').date()

    @traced_function
    def train_pricing(self, origin, destination, departure_date):
        # type: (str, str, date) -> dict
        """
        Получение списка электричек и поездов с указанием цен
        :param origin: express-код станции отправления
        :param destination: express-код станции прибытия
        :param departure_date: дата отправления
        """
        try:
            response = self.post(
                'Railway/V1/Search/TrainPricing',
                json={
                    'CarGrouping': 'Group',
                    'Origin': origin,
                    'Destination': destination,
                    'DepartureDate': departure_date.strftime('%Y-%m-%dT00:00:00')
                }
            )
            return response.json()

        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return {'Trains': []}
            else:
                raise

    @traced_function
    def order_list(self, orders_date):
        # type: (date) -> dict
        """Получение информации о заказах за указанную дату"""
        try:
            response = self.post(
                'Order/V1/Info/OrderList',
                json={'Date': orders_date.strftime('%Y-%m-%dT00:00:00')},
                timeout=600
            )
            return response.json()

        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return {'Orders': []}
            else:
                raise
