# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import AnyStr


from travel.library.python.base_http_client import BaseHttpClient, OAuthHeaderCreator
from travel.rasp.library.python.api_clients.infra.models import Services

log = logging.getLogger(__name__)


class InfraClient(BaseHttpClient):
    HTTP_CLIENT_NAME = 'infra.yandex-team.ru'
    TIMEOUT = 20
    DISABLE_RETRY_CONFIG = True
    DISABLE_CIRCUIT_BREAKER_CONFIG = True

    def __init__(self, host='https://infra-api.yandex-team.ru', auth_token=None, **kwargs):
        self.auth_token = auth_token
        super(InfraClient, self).__init__(
            host=host,
            custom_headers_creators=[OAuthHeaderCreator(self.auth_token)]
            , **kwargs)

    def get_namespaces(self):
        # type: () -> dict
        return self.get('/v1/namespaces').json()

    def get_namespace_by_name(self, namespace):
        # type: (AnyStr) -> dict
        for namespace_data in self.get_namespaces():
            if namespace.lower() == namespace_data['name'].lower():
                return namespace_data

    def get_namespace_services(self, namespace_id):
        # type: (int) -> dict
        return self.get('/v1/namespace/{}/services/'.format(namespace_id)).json()

    def get_environments(self):
        # type: () -> dict
        return self.get('/v1/environments').json()

    def create_service(self, name, namespace_id, abc):
        # type: (AnyStr, int, AnyStr) -> dict
        return self.post('/v1/services', data={
            "name": name,
            "namespaceId": namespace_id,
            "abc": abc,
            "description": "{} for {}".format(name, namespace_id),
        }).json()

    def create_environment(self, name, service_id):
        # type: (AnyStr, int) -> dict
        return self.post('/v1/environments', data={
            "name": name,
            "serviceId": service_id,
            "man": "true",
            "myt": "true",
            "sas": "true",
            "vla": "true",
            "iva": "true",
        }).json()

    def get_services_and_envs_for_namespace(self, namespace_id):
        # type: (int) -> Services
        services = Services()
        for service_data in self.get_namespace_services(namespace_id):
            services.add(service_data['name'], service_data['id'])

        for env in self.get_environments():
            service_name = services.get(env['service_id'])
            if service_name:
                service_name.add_env(env['name'], env['id'])

        return services
