# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import date, datetime
from typing import AnyStr, Dict, List
from decimal import Decimal

import requests

from travel.library.python.tracing.instrumentation import traced_function
from travel.library.python.base_http_client import BaseHttpClient, CircuitBreakerConfig, RetryConfig


log = logging.getLogger(__name__)


class MovistaClient(BaseHttpClient):
    """
    Клиент для работы с АПИ Мовисты - партнера для продаж билетов ЦППК
    https://st.yandex-team.ru/RASPFRONT-8718
    """
    HTTP_CLIENT_NAME = 'Movista'
    RETRY_CONFIG = RetryConfig(total=2)
    CIRCUIT_BREAKER_CONFIG = CircuitBreakerConfig(fail_max=2, reset_timeout=20)
    CANCELS_TIMEOUT = 10
    REPORT_TIMEOUT = 20

    def __init__(self, host, api_token, **kwargs):
        headers = kwargs.pop('headers', {})
        headers.update({'token': api_token})
        super(MovistaClient, self).__init__(host, headers=headers, **kwargs)

    def dt_to_movista_date(self, dt):
        # type: (datetime) -> unicode
        return dt.strftime('%Y-%m-%d')

    def dt_to_movista_dt(self, dt):
        # type: (datetime) -> unicode
        return dt.strftime('%Y-%m-%dT%H:%M:%S')

    def date_from_movista_date(self, dtstr):
        # type: (unicode) -> date
        return datetime.strptime(dtstr, '%Y-%m-%d').date()

    def movista_price_to_decimal(self, price):
        """
        Мовиста отдает price в json как число, а не как строку, поэтому json достет ее как float, что автоматически вносит ошибку:

        f = 48.99
        Decimal(f)  # -> Decimal("48.99000000000000198951966012828052043914794921875")

        Считаем, что имеется в виду сумма с точностью до двух знаков, поэтому явным образом преобразуем к такой точности.
        Это можно сделать только через строку, т.к. при передаче float не помогут никакие округления.
        """
        return Decimal('{:.2f}'.format(price))

    def call_and_parse(self, endpoint, json=None, timeout=None):
        try:
            response = self.post(
                'api/v1/{}'.format(endpoint),
                json=json,
                timeout=timeout
            )
            return response.json()
        except requests.HTTPError as ex:
            if ex.response.status_code == 404:
                return None
            else:
                raise

    @traced_function
    def cancels(self, date):
        # type: (datetime) -> List
        """
        Пример запроса
        curl -XPOST -H "Content-type: application/json" -H "token: token" \
        --data '{"date": "2021-03-25"}' "https://api-6k.test.cppktrain.ru/api/v1/timetable/cancels"

        Пример ответа:
        [
            {
                "createDate": "2021-03-25T19:19:33.513",
                "date": "2021-03-25",
                "trainNumber": "6068",
                "startExpressId": 2000002,
                "finishExpressId": 2000035,
                "fromExpressId": 2000002,
                "toExpressId": 2000035
            }
        ]
        createDate - дата обновления
        date - дата начала движения поезда
        trainNumber - номер поезда
        startExpressId - начальная станция движения поезда
        finishExpressId - конечная станция движения поезда
        fromExpressId - станция, с которой отменили движение
        toExpressId - последняя станция маршрута поезда, на которой не предполагается остановки
        (если отмена частичная, то toExpressId не включается в интервал, на ней поезд останавливается.)
        """
        return self.call_and_parse(
            'timetable/cancels',
            json={'date': self.dt_to_movista_date(date)},
            timeout=self.CANCELS_TIMEOUT
        )

    @traced_function
    def fares(self, date, from_express_id, to_express_id):
        # type: (datetime, unicode, unicode) -> Dict
        """
        Пример запроса
        curl -v -XPOST -H "token: <token>" -H "Content-type: application/json" \
        --data '{"fromExpressId": 2000225, "toExpressId": 2000065, "date": "2020-11-24"}' "https://api-6k.test.cppktrain.ru/api/v1/fares"

        Пример ответа
        {
            "fares": [
                {
                    "price": 96.0,
                    "fareId": 250436,
                    "fromExpressId": 2000225,
                    "toExpressId": 2000065,
                    "farePlan": "ЭКСПРЕСС",
                    "ticketType": "Разовый полный"
                },
                {
                    "price": 48.0,
                    "fareId": 40371,
                    "fromExpressId": 2000225,
                    "toExpressId": 2000065,
                    "farePlan": "Пассажирский",
                    "ticketType": "Разовый полный"
                }
            ],
            "sale": true
        }

        """
        result = self.call_and_parse(
            'fares',
            json={
                'date': self.dt_to_movista_date(date),
                'fromExpressId': from_express_id,
                'toExpressId': to_express_id,
            }
        )
        if result is None:
            return {'fares': [], 'sale': False}
        return result

    @traced_function
    def timetable(self, date, from_express_id, to_express_id):
        # type: (datetime, unicode, unicode) -> Dict
        result = self.call_and_parse(
            'timetable',
            json={
                'date': self.dt_to_movista_date(date),
                'fromExpressId': from_express_id,
                'toExpressId': to_express_id,
            }
        )
        if result is None:
            return {}
        return result

    @traced_function
    def stops(self):
        # type: () -> List
        return self.call_and_parse('stops')

    @traced_function
    def activation(self, order_id, dt, is_success, qr_body, ticket_body):
        # type: (int, datetime, bool, AnyStr, AnyStr) -> Dict
        return self.call_and_parse(
            'activation',
            json={
                'orderId': order_id,
                'date': self.dt_to_movista_dt(dt),
                'success': is_success,
                'qrBody': qr_body,
                'ticketBody': ticket_body
            }
        )

    @traced_function
    def report(self, from_dt, to_dt=None, status=None):
        # type: (datetime, datetime, AnyStr) -> Dict

        if to_dt:
            request = {
                'date': self.dt_to_movista_dt(from_dt),
                'date2': self.dt_to_movista_dt(to_dt)
            }
        else:
            request = {'date': self.dt_to_movista_date(from_dt)}

        if status:  # 'confirmed' or 'refunded'
            request['status'] = status

        return self.call_and_parse(
            'report',
            json=request,
            timeout=self.REPORT_TIMEOUT
        )
