# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import subprocess

from typing import AnyStr, List, Dict
from urllib3.util.retry import Retry

from travel.library.python.base_http_client import BaseHttpClient, OAuthHeaderCreator, RetryConfig

log = logging.getLogger(__name__)


class SandboxClient(BaseHttpClient):
    RETRY_CONFIG = RetryConfig(total=3, method_whitelist=Retry.DEFAULT_METHOD_WHITELIST | {'POST'})
    TIMEOUT = 20
    DISABLE_CIRCUIT_BREAKER_CONFIG = True
    HTTP_CLIENT_NAME = 'Sandbox'

    def __init__(self, host, auth_token=None, **kwargs):
        self.auth_token = auth_token
        super(SandboxClient, self).__init__(
            host=host,
            custom_headers_creators=[OAuthHeaderCreator(self.auth_token)],
            **kwargs)

    def create_task_draft(self, task_type, **kwargs):
        # type: (AnyStr, Dict) -> int

        args = {'type': task_type}
        args.update(kwargs)

        log.info('Create task draft with params %r', args)
        response = self.post('task', json=args)
        task_id = response.json()['id']
        log.info('Task draft #%d has created.', task_id)

        return task_id

    def start_task(self, task_id):
        # type: (int) -> None

        log.info('Starting task #%d.', task_id)
        self.put('batch/tasks/start', json=[task_id])
        log.info('Task #%d has started.', task_id)

    def get_task_status(self, task_id):
        # type: (int) -> AnyStr

        log.info('Get Task #%d status.', task_id)
        response = self.get('task/{}'.format(task_id))
        status = response.json()['status']
        log.info('Task #%d status is %s.', task_id, status)

        return status

    def get_task_resources(self, task_id):
        # type: (int) -> List[Dict]

        log.info('Getting Task #%d resources.', task_id)
        response = self.get('task/{}/resources'.format(task_id))
        resources = response.json().get('items', [])
        log.info('Received Task #%d resources.', task_id)

        return resources

    def get_resource(self, resource_type, latest=True, order=None, state=None, attrs=None):
        # type: (str, bool, str, str, Dict) -> Dict

        if latest:
            order = '-id'

        return self.get_resources(resource_type=resource_type, limit=1, order=order, state=state, attrs=attrs)[0]

    def get_resources(self, resource_type, limit, order=None, state=None, attrs=None):
        # type: (str, int, str, str, Dict) -> List[Dict]

        params = {'type': resource_type}
        if limit:
            params['limit'] = limit
        if order:
            params['order'] = order
        if state:
            params['state'] = state
        if attrs:
            params['attrs'] = json.dumps(attrs)

        response = self.get('resource', params=params)
        resources = response.json().get('items', [])
        if not resources:
            raise Exception('Resources not found. params: {}'.format(params))

        return resources

    def upload_file(self, resource_type, target, ttl='inf', attrs=None):
        # type: (str, str, str, Dict) -> None

        attrs_list = ['--attr {key}={value}'.format(key=k, value=v) for k, v in attrs.items()]
        attrs_str = ' '.join(attrs_list)

        cmd = 'ya upload --type={resource_type} --owner=RASP --token={{token}} --ttl={ttl} {attrs} {target}'.format(
            resource_type=resource_type,
            ttl=ttl,
            attrs=attrs_str,
            target=target
        )

        try:
            out = subprocess.check_output(
                cmd.format(token=self.auth_token),
                shell=True,
                stderr=subprocess.STDOUT,
            )
            log.debug(out)

        except subprocess.CalledProcessError as exc:
            log.error('%s ya upload failed. returncode %s\n%s' % (resource_type, exc.returncode, exc.output))
