# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import requests

from travel.library.python.base_http_client import BaseHttpClient, OAuthHeaderCreator, RetryConfig


log = logging.getLogger(__name__)


class SandboxProxyClient(BaseHttpClient):
    RETRY_CONFIG = RetryConfig(total=3)
    HTTP_CLIENT_NAME = 'SandboxProxy'

    def __init__(self, host, oauth_token, **kwargs):
        super(SandboxProxyClient, self).__init__(
            host=host,
            custom_headers_creators=[OAuthHeaderCreator(oauth_token)],
            **kwargs
        )

    def get_sandbox_proxy_response(self, path, params=None, timeout=None):
        """
        Запрос к Sandbox HTTP-proxy
        :param path: добавка для образования урла ручки
        :param params: словарь GET-параметров
        :param timeout: допустимое время ответа ручки в секундах
        :return: ответ в формате JSON
        """

        params = params or {}

        try:
            response = self.get(path, params=params, timeout=timeout)
        except (requests.ConnectionError, requests.Timeout) as ex:
            log.exception('Sandbox HTTP-proxy request error. {}'.format(repr(ex)))
            raise

        try:
            response.raise_for_status()
        except requests.HTTPError:
            log.exception('Sandbox HTTP-proxy response bad status. {}'.format(response.request.body))
            raise

        log.info('Request: {}, response code: {}'.format(response.request.url, response.status_code))
        return response

    def get_last_resource(self, resource_type):
        """Получение последней версии sandbox-ресурса"""
        return self.get_sandbox_proxy_response('last/{}'.format(resource_type))
