# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function


class Status(object):
    STATUS_SUCCESS = 'success'
    STATUS_FAIL = 'fail'
    STATUS_ERROR = 'error'


class Fail(Exception):
    """
    There was a problem with the data submitted, or some pre-condition of the API call wasn't satisfied

    Attributes:
        data -- Details of why the request failed. If the reasons for failure correspond to POST values,
                the response object's keys SHOULD correspond to those POST values.
    """
    def __init__(self, data):
        super(Fail, self).__init__("fail")
        self.data = data


class Error(Exception):
    """
    An error occurred in processing the request, i.e. an exception was thrown

    Attributes:
        message -- A meaningful, end-user-readable (or at the least log-worthy) message, explaining what went wrong.
        code -- A numeric code corresponding to the error, if applicable.
        data -- A generic container for any other information about the error, i.e. the conditions that caused
                the error, stack traces, etc.
    """
    def __init__(self, message, code, data):
        super(Error, self).__init__(message)
        self.code = code
        self.data = data


class InvalidResponse(Exception):
    pass


def process_errors(response_json, status_code):
    if 'status' not in response_json:
        raise InvalidResponse("JSend response doesn't have status field")
    status = response_json['status']

    if status == Status.STATUS_ERROR:
        if 'message' not in response_json:
            raise InvalidResponse("JSend error response doesn't have message field")
        raise Error(response_json['message'], response_json.get('code'), response_json.get('data'))

    if 'data' not in response_json:
        raise InvalidResponse("JSend response doesn't have data field")

    if status == Status.STATUS_FAIL:
        if status_code >= 499:
            raise Fail(response_json['data'])


def read(response_json):
    status = response_json['status']

    if status == Status.STATUS_SUCCESS:
        return response_json['data']

    if status == Status.STATUS_FAIL:
        raise Fail(response_json['data'])

    raise InvalidResponse('Invalid JSend response status {!r}'.format(status))
