# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from typing import AnyStr, Optional, Dict

import requests

from travel.library.python.tracing.instrumentation import traced_function
from travel.library.python.base_http_client import BaseHttpClient


log = logging.getLogger(__name__)


class OrderState(object):
    IN_PROGRESS = 'IN_PROGRESS'
    NEW = 'NEW'
    RESERVED = 'RESERVED'
    WAITING_PAYMENT = 'WAITING_PAYMENT'
    CONFIRMED = 'CONFIRMED'
    CANCELLED = 'CANCELLED'
    PAYMENT_FAILED = 'PAYMENT_FAILED'
    REFUNDED = 'REFUNDED'

    ALL = {IN_PROGRESS, NEW, RESERVED, WAITING_PAYMENT, CONFIRMED, CANCELLED, PAYMENT_FAILED, REFUNDED}


def get_travel_api_order_state(api_state, default_state):
    # type: (AnyStr, AnyStr) -> AnyStr
    # TODO: Специфические для бэкенда внутренние состояния превращаем в поддерживаемые фронтом
    # Уберем, когда сделают https://st.yandex-team.ru/TRAVELBACK-1918 и https://st.yandex-team.ru/TRAVELBACK-1934
    return api_state if api_state in OrderState.ALL else default_state


class TravelApiUserIdent(object):
    """Класс с информацией об идентификации пользователя для Travel API"""
    def __init__(
        self, session_id=None, yandex_uid=None, user_ticket=None, blackbox_user_uid=None,
        device_id=None, user_agent=None, user_ip=None
    ):
        # type: (Optional[AnyStr], Optional[AnyStr], Optional[AnyStr], Optional[AnyStr], Optional[AnyStr], Optional[AnyStr], Optional[AnyStr]) -> None

        self.session_id = session_id
        self.yandex_uid = yandex_uid
        self.user_ticket = user_ticket
        self.blackbox_user_uid = blackbox_user_uid
        self.device_id = device_id
        self.user_agent = user_agent
        self.user_ip = user_ip


class TravelApiClient(BaseHttpClient):
    """
    Клиент для работы с заказами через Travel API
    Форматы ручек API: https://api.travel-balancer-test.yandex.net/swagger-ui.html#/generic-booking-flow-controller
    """
    HTTP_CLIENT_NAME = 'Travel API'
    DISABLE_CIRCUIT_BREAKER_CONFIG = True

    def __init__(self, host, tvm_header_creator, create_order_timeout=10, does_city_static_page_exist_circuit_breaker_config=None, **kwargs):
        custom_headers_creators = list(kwargs.pop('custom_headers_creators', []))
        custom_headers_creators.append(tvm_header_creator)

        super(TravelApiClient, self).__init__(
            host=host,
            custom_headers_creators=custom_headers_creators,
            **kwargs
        )
        self._create_order_retry_config = self.get_instance_retry_config().evolve(method_whitelist={'POST'})
        self._create_order_timeout = create_order_timeout
        self._does_city_static_page_exist_circuit_breaker_config = does_city_static_page_exist_circuit_breaker_config

    def make_headers(self, user_ident):
        # type: (TravelApiUserIdent) -> Dict[AnyStr, AnyStr]

        headers = {'Content-Type': 'application/json'}
        if user_ident.session_id:
            headers['X-Ya-Session-Key'] = user_ident.session_id
        if user_ident.yandex_uid:
            headers['X-Ya-YandexUid'] = user_ident.yandex_uid
        if user_ident.user_ticket:
            headers['X-Ya-User-Ticket'] = user_ident.user_ticket
        if user_ident.blackbox_user_uid:
            headers['X-Ya-PassportId'] = user_ident.blackbox_user_uid
        if user_ident.user_agent:
            headers['X-Ya-User-Agent'] = user_ident.user_agent
        if user_ident.user_ip:
            headers['X-Ya-User-Ip'] = user_ident.user_ip

        return headers

    def call_and_parse(
        self, url_path, user_ident, params=None, request_content=None, has_json_response=True,
        timeout=None, retry_config=None, disable_retry_config=None, circuit_breaker_config=None
    ):
        http_method = 'POST' if request_content else 'GET'
        headers = self.make_headers(user_ident)
        try:
            response = self.make_request(
                http_method,
                url_path,
                params=params,
                json=request_content,
                headers=headers,
                timeout=timeout,
                retry_config=retry_config,
                disable_retry_config=disable_retry_config,
                circuit_breaker_config=circuit_breaker_config
            )
        except requests.HTTPError:
            raise

        try:
            result = response.json() if has_json_response else None
        except Exception:
            log.exception('Error in Travel API response json', response.content)
            raise

        return result

    @traced_function
    def create_order(self, user_ident, request_content):
        # type: (TravelApiUserIdent, Dict) -> Dict
        """Создание заказа"""

        api_response = self.call_and_parse(
            'generic_booking_flow/v1/create_order',
            user_ident,
            request_content=request_content,
            retry_config=self._create_order_retry_config,
            timeout=self._create_order_timeout
        )
        return api_response

    @traced_function
    def get_order(self, user_ident, order_uid):
        # type: (TravelApiUserIdent, AnyStr) -> Dict
        """Подробная информация о заказе"""

        api_response = self.call_and_parse(
            'generic_booking_flow/v1/get_order',
            user_ident,
            params={'orderId': order_uid}
        )
        return api_response

    @traced_function
    def get_order_state(self, user_ident, order_uid):
        # type: (TravelApiUserIdent, AnyStr) -> Dict
        """Состояние заказа"""

        api_response = self.call_and_parse(
            'generic_booking_flow/v1/get_order_state',
            user_ident,
            params={'orderId': order_uid}
        )
        return api_response

    @traced_function
    def get_order_state_batch(self, user_ident, request_content):
        api_response = self.call_and_parse(
            'generic_booking_flow/v1/get_order_state_batch',
            user_ident,
            request_content=request_content
        )
        return api_response

    @traced_function
    def start_payment(self, user_ident, request_content):
        # type: (TravelApiUserIdent, Dict) -> Dict
        """Запуск оплаты"""

        api_response = self.call_and_parse(
            'generic_booking_flow/v1/start_payment',
            user_ident,
            request_content=request_content,
            has_json_response=False
        )
        return api_response

    @traced_function
    def get_hotel_city_static_page(self, user_ident, slug, domain='ru'):
        """Данные для рендеринга статической страницы отелей"""

        params = {'domain': domain, 'regionSlug': slug}

        api_response = self.call_and_parse(
            'hotels_portal/v1/city_static_page',
            user_ident,
            params=params
        )

        return api_response

    @traced_function
    def does_hotel_city_static_page_exist(self, user_ident, geo_id):
        """Проверка существования лендинга отелей"""

        params = {'geo_id': geo_id}

        api_response = self.call_and_parse(
            'hotels_portal/v1/does_city_static_page_exist',
            user_ident,
            params=params,
            disable_retry_config=True,
            circuit_breaker_config=self._does_city_static_page_exist_circuit_breaker_config
        )

        if api_response and 'exists' in api_response:
            return api_response.get('exists')

        return None
