# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from itertools import chain

from django.conf import settings
from pybreaker import CircuitBreaker
from pymongo import ReadPreference
from mongoengine import DoesNotExist, Q

from common.apps.archival_data.models import ArchivalSearchData, ArchivalSettlementsData
from common.models.geo import Settlement
from common.models.transport import TransportType
from common.settings.utils import define_setting
from travel.library.python.tracing.instrumentation import traced_function

log = logging.getLogger(__name__)

# флаг для тестирования
define_setting('ARCHIVAL_DATA_SHOW_ALWAYS', default=False, converter=lambda x: bool(x))

define_setting('ARCHIVAL_DATA_BREAKER_PARAMS', default={'fail_max': 5, 'reset_timeout': 20})
archival_breaker = CircuitBreaker(**settings.ARCHIVAL_DATA_BREAKER_PARAMS)


def get_canonical(point_from, point_to, transport_types):
    return {'point_from': point_from,
            'point_to': point_to,
            'transport_type': transport_types[0] if len(transport_types) == 1 else None}


@archival_breaker
@traced_function
def get_archival_data(point_from, point_to, valid_transport_types=None, add_segments=True):
    query = ArchivalSearchData.objects.read_preference(ReadPreference.SECONDARY_PREFERRED).filter(
        point_from=point_from.point_key,
        point_to=point_to.point_key
    )

    raw_data = list(query.aggregate())

    if not raw_data:
        return None

    transport_type_ids = {entry['transport_type'] for entry in raw_data}
    transport_types = [t.code for t in TransportType.objects.filter(id__in=transport_type_ids)]

    if valid_transport_types:
        transport_types = [t for t in transport_types if t in valid_transport_types]

        if not transport_types:
            return None

    segments = get_archival_segments(point_from, point_to, valid_transport_types) if add_segments else None

    return {
        'canonical': get_canonical(point_from, point_to, transport_types),
        'transport_types': transport_types,
        'segments': segments
    }


def get_extended_settlements(point_from, point_to, valid_transport_types):
    from geosearch.views.pointtopoint import iter_extended_points

    for ext_point_from, ext_point_to in iter_extended_points(
        point_from,
        point_to,
        valid_transport_types == {'suburban'}
    ):
        if isinstance(ext_point_from, Settlement) and isinstance(ext_point_to, Settlement):
            yield ext_point_from, ext_point_to


@archival_breaker
@traced_function
def get_archival_segments(point_from, point_to, valid_transport_types=None):
    if isinstance(point_from, Settlement) and isinstance(point_to, Settlement):
        settlements = [(point_from, point_to)]
    else:
        settlements = get_extended_settlements(point_from, point_to, valid_transport_types)

    settlements_data = None
    for settlement_from, settlement_to in settlements:
        try:
            query = Q(point_from=settlement_from.point_key, point_to=settlement_to.point_key)

            if valid_transport_types:
                query &= Q(transport_type__in=[t.id for t in TransportType.objects.filter(code__in=valid_transport_types)])

            settlements_data = ArchivalSettlementsData.objects.read_preference(ReadPreference.SECONDARY_PREFERRED).filter(query)
            break
        except DoesNotExist:
            log.debug('DoesNotExist {} {}'.format(point_from.point_key, point_to.point_key))

    if not settlements_data:
        return

    raw_data = settlements_data
    segments = list(chain(*[data.to_mongo().to_dict()['segments'] for data in raw_data]))

    filtered_segments = []
    for segment in segments:
        if (
            (isinstance(point_from, Settlement) or segment['station_from']['id'] == point_from.id)
            and (isinstance(point_to, Settlement) or segment['station_to']['id'] == point_to.id)
        ):
            filtered_segments.append(segment)

    return filtered_segments if filtered_segments else None
