# coding: utf8
from collections import defaultdict

from common.apps.facility.models import SuburbanThreadFacility
from common.models.transport import TransportType
from common.utils.date import RunMask


def fill_suburban_facilities(facilitable_objects, start_date_getter):
    """
    Устанавливаем на объекты из facilitable_objects их удобства (suburban_facilities),
    используя thread и start_date с этих объектов.

    :type facilitable_objects: [RThreadSegment, ScheduleRoute]
    :type start_date_getter: callable
    :return: None
    """
    facilitable_objects = [s for s in facilitable_objects
                           if s.thread and s.thread.t_type_id == TransportType.SUBURBAN_ID]

    thread_ids = [s.thread.id for s in facilitable_objects]

    thread_facilities_by_thread = defaultdict(list)
    for thread_facility in (SuburbanThreadFacility.objects.filter(thread__id__in=thread_ids)
                            .prefetch_related('facilities')):
        thread_facilities_by_thread[thread_facility.thread_id].append(thread_facility)

    for s in facilitable_objects:
        for thread_facility in thread_facilities_by_thread.get(s.thread.id, []):
            if RunMask.runs_at(thread_facility.year_days, start_date_getter(s)):
                s.suburban_facilities = sorted(thread_facility.facilities.all(), key=lambda x: x.order)


def get_suburban_thread_facilities(thread_id, start_date):
    try:
        thread_facility = SuburbanThreadFacility.objects.prefetch_related('facilities').get(thread_id=thread_id)
    except SuburbanThreadFacility.DoesNotExist:
        return None
    return (
        None if not RunMask.runs_at(thread_facility.year_days, start_date)
        else sorted(thread_facility.facilities.all(), key=lambda x: x.order)
    )
