# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
from datetime import timedelta

import yt.wrapper

from common.data_api.yt.instance import update_yt_wrapper_config
from common.data_api.yt.yt_lib import get_all_yt_table_rows


log = logging.getLogger(__name__)


YT_IM_CARPRICING_TABLES_DIR = '//home/logfeller/logs/rasp-im-carpricing-log/1d/'
YT_IM_TRAIN_CAR_TYPES_TABLES_DIR = '//home/rasp/rasp/im_train_car_types/'

FIELDS_MAPPING = {
    'TrainInfo__TrainName': 'train_name',
    'Car__TrainNumber': 'train_number',
    'Car__CarSubType': 'car_sub_type'
}


def im_reducer(key, recs):
    result = {FIELDS_MAPPING[field]: key[field] for field in FIELDS_MAPPING.keys()}
    result['im_records_count'] = sum(1 for _ in recs)
    yield result


def group_im_table(table_date):
    """
    Группировка данных таблицы лога ИМ в маленькую таблицу
    :param table_date: дата таблицы лога
    """
    table_name = table_date.strftime('%Y-%m-%d')
    source_table = '{}{}'.format(YT_IM_CARPRICING_TABLES_DIR, table_name)
    sorted_table = '{}{}'.format(YT_IM_TRAIN_CAR_TYPES_TABLES_DIR, 'tmp-im-carpricing-sorted')
    destination_table = '{}{}'.format(YT_IM_TRAIN_CAR_TYPES_TABLES_DIR, table_name)

    update_yt_wrapper_config()

    if not yt.wrapper.exists(source_table):
        log.error('Table {} does not exists'.format(sorted_table))
        raise Exception('Table {} does not exists'.format(sorted_table))

    log.info('Make {} from {}'.format(sorted_table, source_table))
    yt.wrapper.run_sort(
        source_table,
        sorted_table,
        sort_by=FIELDS_MAPPING.keys()
    )

    log.info('Reduce {} to {}'.format(sorted_table, destination_table))
    yt.wrapper.run_reduce(
        im_reducer,
        sorted_table,
        destination_table,
        reduce_by=list(FIELDS_MAPPING.keys())
    )


class ImTrainCarType(object):
    """
    Объект, соответствующий элементам лога ИМ, сгруппированным по типам поезов и вагонов
    """
    def __init__(self, train_name, train_number, car_sub_type, im_records_count):
        self.train_name = train_name
        self.wagon = car_sub_type
        self.train_number = train_number
        self.records_count = im_records_count


def im_train_car_type_from_yt_row(yt_row):
    return ImTrainCarType(
        yt_row['train_name'].decode('utf-8'),
        yt_row['train_number'].decode('utf-8'),
        yt_row['car_sub_type'].decode('utf-8'),
        yt_row['im_records_count']
    )


def get_im_train_car_types(date_from, date_to):
    """
    Возвращает итератор по таблицам сгруппированных логов ИМ
    Используем данные из таблиц за указанные даты
    """
    update_yt_wrapper_config()

    day = date_to
    while day >= date_from:
        im_table = '{}{}'.format(YT_IM_TRAIN_CAR_TYPES_TABLES_DIR, day.strftime('%Y-%m-%d'))
        for row in get_all_yt_table_rows(im_table):
            try:
                im_train_car_type = im_train_car_type_from_yt_row(row)
            except Exception as ex:
                log.warning(ex.message)
                continue
            yield im_train_car_type

        day -= timedelta(days=1)
