# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import warnings

from django.conf import settings
from django.db import connection
from django.utils.translation import get_language
from mongoengine import Q
from pybreaker import CircuitBreaker

from common.models.geo import Settlement, Station, ExternalDirection
from common.models.teasers import Page
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.warnings import RaspDeprecationWarning
from travel.library.python.tracing.instrumentation import traced_function

from .models import Info, ObjLinkType


log = logging.getLogger(__name__)

define_setting('INFO_CENTER_BREAKER_PARAMS', default={'fail_max': 2, 'reset_timeout': 120})

info_center_breaker = CircuitBreaker(**settings.INFO_CENTER_BREAKER_PARAMS)


class _LinkedObjectsList(list):
    def add(self, obj_type, obj_key):
        self.append((obj_type, obj_key))

    def to_query(self):
        return Q(__raw__={
            '$or': [  # ищем info, к которым привязан хотя бы один интересующий нас объект
                {'linked_objects': {
                    '$elemMatch': {'$elemMatch': {
                        'obj_type': obj_type,
                        'obj_key': obj_key,
                    }}
                }}
                for obj_type, obj_key in self
            ],
        })


class InfoGetter(object):
    def __init__(self):
        self.linked_objects = _LinkedObjectsList()

    @info_center_breaker
    @traced_function
    def get(self, services, page, national_versions=None, lang=None, data=None, additional_query=None):
        today = environment.today()
        query = Q(
            dt_from__lte=today,
            dt_to__gte=today,
            services__in=services,
            lang=lang or get_language() or 'ru',
            national_versions__in=national_versions or [self._get_national_version()],
        )

        query &= self._get_page_query(page, data)
        query &= additional_query

        return list(Info.objects.filter(query))

    def _get_page_query(self, page, data):
        """ Собираем запрос по связанным объектам. """
        self.linked_objects = _LinkedObjectsList()

        self.linked_objects.add(ObjLinkType.PAGE, Page.ALL)
        self.linked_objects.add(ObjLinkType.PAGE, page)

        # для любой страницы может быть специальный обработчик связанных объектов
        page_linked_objects = getattr(self, '_page_linked_objects_' + page, None)
        if page_linked_objects:
            page_linked_objects(data)

        return self.linked_objects.to_query()

    def _page_linked_objects_info_settlement(self, settlement):
        self._add_obj_links(settlement)

    def _page_linked_objects_info_station(self, station):
        self._add_station(station)

    def _page_linked_objects_search(self, data):
        from_to_points, segments = data['points'], data['routes']

        self._add_obj_links(from_to_points)
        self._add_ext_directions(from_to_points)
        self._add_stations_pairs(segments)

    def _page_linked_objects_search_suburban(self, data):
        """
        Логика работы как у search, но оставляем отдельную страницу
        'search_suburban', чтобы можно было её указывать в привязках Page.
        """
        return self._page_linked_objects_search(data)

    def _page_linked_objects_direction(self, ext_direction):
        self._add_obj_links(ext_direction)

    def _add_obj_links(self, objs):
        if not isinstance(objs, (list, tuple)):
            objs = [objs]

        for obj in objs:
            if isinstance(obj, Station):
                self.linked_objects.add(ObjLinkType.STATION, obj.id)
            elif isinstance(obj, Settlement):
                self.linked_objects.add(ObjLinkType.SETTLEMENT, obj.id)
            elif isinstance(obj, ExternalDirection):
                self.linked_objects.add(ObjLinkType.DIRECTION, obj.id)

    def _add_station(self, station):
        self._add_obj_links(station)

        external_directions = [
            marker.external_direction
            for marker in station.externaldirectionmarker_set
                                 .all().select_related('external_direction')
        ]
        self._add_obj_links(external_directions)

    def _add_ext_directions(self, from_to_points):
        point_from, point_to = from_to_points
        if isinstance(point_from, Station) and isinstance(point_to, Station):
            dirs1 = [marker.external_direction for marker in
                     point_from.externaldirectionmarker_set.all().select_related(
                         'external_direction')]
            dirs2 = [marker.external_direction for marker in
                     point_to.externaldirectionmarker_set.all().select_related(
                         'external_direction')]

            if len(dirs2) < len(dirs1):
                dirs1, dirs2 = dirs2, dirs1

            external_directions = [d for d in dirs1 if d in dirs2]
            self._add_obj_links(external_directions)

    def _add_stations_pairs(self, segments):
        if not segments:
            return

        from_to_stations_pairs = list(set(
            (segment.station_from.id, segment.station_to.id)
            for segment in segments if not getattr(segment, 'is_transfer_segment', False)
        ))

        sql = '''
            SELECT external_direction_id
            FROM www_externaldirectionmarker
            WHERE station_id in (%d, %d)
            GROUP BY external_direction_id
            HAVING count(station_id) >= 2'''

        if from_to_stations_pairs:
            res_sql = sql % from_to_stations_pairs[0]

            for station_pair in from_to_stations_pairs[1:]:
                res_sql += ' UNION ' + sql % station_pair

            cursor = connection.cursor()
            cursor.execute(res_sql)

            ids = [row[0] for row in cursor.fetchall()]
            for direction_id in ids:
                self.linked_objects.add(ObjLinkType.DIRECTION, direction_id)

    def _get_national_version(self):
        warnings.warn(
            "[2017-08-04] You should manually pass national_version to getter",
            RaspDeprecationWarning, stacklevel=3
        )
        request = environment.get_request()
        return (getattr(request, 'NATIONAL_VERSION', None) if request else None) or 'ru'


def get_info(*args, **kwargs):
    try:
        return InfoGetter().get(*args, **kwargs)
    except Exception:
        log.exception("Unable to retrieve infos")
        return []
