# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import uuid

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from mongoengine import (
    EmbeddedDocument, IntField, StringField, ListField, EmbeddedDocumentListField, DateTimeField,
    DynamicField, DictField, BooleanField
)
from six import python_2_unicode_compatible

from common.db.mongo.base import RaspDocument
from common.db.mongo.counter import MongoCounter
from common.models.geo import Station, Settlement, ExternalDirection
from common.models.teasers import Page, Teaser
from common.utils import uuid64_int


class ObjLinkType(object):
    STATION = 'station'
    SETTLEMENT = 'settlement'
    DIRECTION = 'externaldirection'
    PAGE = 'page'


class ObjectLink(EmbeddedDocument):
    """
    Ex:
    {obj_type: 'station', obj_key: 123456}
    {obj_type: 'page', obj_key: 'search_page'}
    """
    meta = {
        'strict': False
    }

    obj_type = StringField()
    obj_key = DynamicField()

    def __eq__(self, other):
        return self.obj_type == other.obj_type and self.obj_key == other.obj_key


class Message(EmbeddedDocument):
    """ Сообщение ленты """
    meta = {
        'strict': False
    }

    text = StringField()
    dt_created = DateTimeField()


class Push(EmbeddedDocument):
    """ Пуш """
    meta = {
        'strict': False
    }

    # Дефолтные политики отправки пушей:
    #   Обычный - 8 штук в день.
    #   Экстренный - 4 штуки в день.
    #
    # Настройка политик: https://st.yandex-team.ru/RASPFRONT-8296#5f718fff40b3a53950d8d30f
    POLICIES = [
        ('suburban_default_device_id', _(u'Обычный')),
        ('suburban_urgent_device_id', _(u'Экстренный')),
    ]

    id = StringField(required=True)
    title = StringField()
    text = StringField()
    url = StringField()
    image_url = StringField()
    policy = StringField(required=True, choices=POLICIES, default=POLICIES[0][0])

    dt_created = DateTimeField(required=True)
    possible_receivers = IntField(null=True)

    linked_objects = ListField(EmbeddedDocumentListField(ObjectLink))

    log = ListField(DictField())
    error = StringField()

    def clean(self):
        if self.id is None:
            self.id = uuid.uuid4().hex


class YadmNewsItem(EmbeddedDocument):
    """Новость для ЯдМ"""
    id = IntField(primary_key=True)
    title = StringField()
    text = StringField()
    importance = IntField()
    dt_created = DateTimeField()

    def clean(self):
        if self.id is None:
            self.id = MongoCounter(settings.YADM_NEWS_ITEM_ID_COUNTER).next_value()


@python_2_unicode_compatible
class Info(RaspDocument):
    meta = {
        'strict': False,
        'indexes': [
            ['dt_from', 'dt_to', 'services', 'lang']
        ],
        'index_background': True,
    }

    class Type(object):
        NORMAL = 'normal'
        AHTUNG = 'ahtung'
        SPECIAL = 'special'
        BANNER = 'banner'

    TYPES = (
        (Type.NORMAL, _('Обычный')),
        (Type.AHTUNG, _('Внимание')),
        (Type.SPECIAL, _('Спецпредложение')),
        (Type.BANNER, _('Баннер')),
    )

    class Service(object):
        WEB = 'web'
        MOBILE_APPS = 'mobile_apps'

    # ссылки на объекты из базы могут быть как по id, так и по любым другим полям (напр, code для Page)
    OBJ_LINKS_CONFIG = {
        ObjLinkType.STATION: {'class': Station},
        ObjLinkType.SETTLEMENT: {'class': Settlement},
        ObjLinkType.DIRECTION: {'class': ExternalDirection},
        ObjLinkType.PAGE: {
            'class': Page,
            'title_getter': lambda p: p.title,
            'id_to_key': lambda page_id: Page.objects.get(id=page_id).code,
            'key_to_id': lambda page_key: Page.objects.get(code=page_key).id,
        },
    }

    # Не можем использовать ObjectId, т.к. в приложениях зашито использование чисел.
    id = IntField(primary_key=True)
    uuid = IntField(unique=True)
    info_url = StringField(unique=True)  # ссылка на морде на это Info

    title = StringField()
    url = StringField()  # ссылка для title

    text = StringField()
    text_short = StringField()

    info_type = StringField(choices=TYPES, default='ahtung')
    importance = IntField(default=0)

    dt_from = DateTimeField()
    dt_to = DateTimeField()

    lang = StringField(default='r')
    national_versions = ListField(StringField(), default=['r'])

    services = ListField(StringField())
    linked_objects = ListField(EmbeddedDocumentListField(ObjectLink))

    messages = EmbeddedDocumentListField(Message)  # лента сообщений
    pushes = EmbeddedDocumentListField(Push)
    yadm_news = EmbeddedDocumentListField(YadmNewsItem)

    def __str__(self):
        return 'Info {}: {}'.format(self.id, self.title).encode('utf8')

    def clean(self):
        if self.id is None:
            self.id = MongoCounter(settings.INFO_CENTER_ID_COUNTER).next_value()

        if self.uuid is None:
            self.uuid = self.generate_uuid()

        if not self.info_url:
            self.info_url = 'info{}'.format(self.id)

    def to_old_teaser(self):
        """ Подержка старого интерфейса модели Teaser. """
        return Teaser(
            id=self.id,
            title=self.title,
            url=self.url,
            content=self.text,
            mobile_content=self.text_short,
            lang=self.lang,
            national_version=self.national_versions[0] if self.national_versions else 'ru',
            mode=self.info_type,
            importance=self.importance,
            is_active_rasp=Info.Service.WEB in self.services,
            is_active_export=Info.Service.MOBILE_APPS in self.services,
            date_start=self.dt_from.date(),
            date_finish=self.dt_to.date(),
            uid64=self.uuid,
        )

    @classmethod
    def get_linked_obj_types(cls):
        return list(cls.OBJ_LINKS_CONFIG.keys())

    @classmethod
    def get_object_by_link(cls, obj_link):
        key_to_id = cls.OBJ_LINKS_CONFIG[obj_link.obj_type].get('key_to_id', lambda key: int(key))
        obj_id = key_to_id(obj_link.obj_key)

        obj_model = cls.OBJ_LINKS_CONFIG[obj_link.obj_type]['class']
        try:
            return obj_model.objects.get(id=obj_id)
        except obj_model.DoesNotExist:
            return None

    @classmethod
    def get_link_by_obj_id(cls, obj_type, obj_id):
        id_to_key = cls.OBJ_LINKS_CONFIG[obj_type].get('id_to_key', lambda o: int(o))  # by default key == id
        obj_link = ObjectLink(obj_type=obj_type, obj_key=id_to_key(obj_id))
        return obj_link

    @classmethod
    def get_linked_obj_title(cls, obj_type, obj):
        title_getter = Info.OBJ_LINKS_CONFIG[obj_type].get('title_getter', lambda o: o.L_title())
        return title_getter(obj)

    @classmethod
    def generate_uuid(cls):
        return uuid64_int()
