# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import traceback

from django.conf import settings
from django.utils.encoding import force_text

from common.data_api.sup.client import get_client, SupClient
from common.dynamic_settings.default import conf
from common.models.geo import Station
from travel.rasp.library.python.common23.date import environment

from .models import Push, Info, ObjLinkType


log = logging.getLogger(__name__)


class PushSender(object):
    class Error(Exception):
        pass

    def __init__(self, info_id, user=None, oauth_token=None):
        self.info_id = info_id
        self.oauth_token = oauth_token
        self.user = user

        self.info = None
        self.push = None

    def send(self, title, text, url, image_url, policy):
        try:
            self.info = Info.objects.get(id=self.info_id)
            self.push = Push(
                title=title,
                text=text,
                url=url,
                image_url=image_url,
                dt_created=environment.now(),
                linked_objects=self.info.linked_objects,
                policy=policy
            )
            self.info.pushes.append(self.push)
            self.info.save()
        except Exception as e:
            msg = "Can't send push for info {}: {}".format(self.info_id, repr(e))
            log.exception(msg)
            raise PushSender.Error(msg)

        try:
            self._send(dry_run=False)
        except Exception as e:
            self.push.error = repr(e)
            self.log('push send error', '', traceback.format_exc())
            raise
        finally:
            self.info.save()

    def _send(self, dry_run):
        kwargs = dict(
            project='suburban',
            image='ic_notification',
            receivers=[self.make_receivers_query()],
            title=self.push.title,
            text=self.push.text,
            image_url=self.push.image_url,
            url=self.push.url,
            device_id_policy=get_device_id_policy(self.push.policy),
            install_id_policy=get_install_id_policy(),
            dry_run=dry_run,
            high_priority=True,  # https://st.yandex-team.ru/SUBURBAN-253
        )

        self.log('sending (dry-run={})'.format(dry_run), kwargs)
        client = get_client(oauth_token=self.oauth_token)
        response_str = client.pushes(**kwargs)
        response = json.loads(response_str)
        self.log('sending ok (dry-run={})'.format(dry_run), response)

        return response

    def make_receivers_query(self):
        """
        Формируем строку вида:
        'tag:(suburban_station==5123 || suburban_station==5567) && (suburban_city==42 || suburban_direction==43)'
        для отправки в СУП.
        """

        push_receivers_groups = []
        for linked_objs_group in self.info.linked_objects:
            if not linked_objs_group:
                continue

            push_receivers = []
            for linked_object in linked_objs_group:
                if linked_object.obj_type == ObjLinkType.STATION:
                    station = Station.objects.get(id=linked_object.obj_key)
                    station_esr = station.get_code('esr')
                    push_receivers.append('suburban_station=={}'.format(station_esr))
                elif linked_object.obj_type == ObjLinkType.SETTLEMENT:
                    push_receivers.append('suburban_city=={}'.format(linked_object.obj_key))
                elif linked_object.obj_type == ObjLinkType.DIRECTION:
                    push_receivers.append('suburban_direction=={}'.format(linked_object.obj_key))

            push_receivers_groups.append(
                '({})'.format(' || '.join(push_receivers))
            )

        if not push_receivers_groups:
            raise ValueError('Invalid push receivers: no objects selected')

        receivers_filter = " && ".join(push_receivers_groups)
        return SupClient.get_app_id_tag(settings.SUBURBAN_APP_IDS) + " && " + receivers_filter

    def log(self, msg, data, error=None):
        log_entry = {'msg': msg, 'data': data, 'user': self.user}
        if error:
            log_entry['error'] = str(error)

        log.info(force_text(log_entry))

        if self.push:
            self.push.log.append(log_entry)
            self.info.save()


def get_dir_pair_key(dir1_id, dir2_id):
    """ Формируем ключ для подписки на пару направлений. """
    sorted_dir_pair = sorted((dir1_id, dir2_id))
    return '{}+{}'.format(sorted_dir_pair[0], sorted_dir_pair[1])


def get_device_id_policy(policy):
    if policy == 'suburban_default_device_id':
        return conf.SUP_DEFAULT_DEVICE_ID_POLICY
    if policy == 'suburban_urgent_device_id':
        return conf.SUP_URGENT_DEVICE_ID_POLICY


def get_install_id_policy():
    return conf.SUP_INSTALL_ID_POLICY
