# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime

from factory.mongoengine import MongoEngineFactory

from common.apps.suburban_events.models import (
    CancelledStation, CompanyCrash, EventState, HourEventsRate, MovistaCancelRaw, StationCancel, StationEvent,
    StationExpectedEvent, StationUpdateInfo, SuburbanKey, ThreadEvents, ThreadKey, ThreadStationKey, ThreadStationState
)
from common.apps.suburban_events.utils import get_rtstation_key, get_thread_suburban_key


class ThreadEventsFactory(MongoEngineFactory):
    class Meta:
        model = ThreadEvents

    @classmethod
    def create(cls, **kwargs):
        thread = kwargs.pop('thread', None)
        thread_start_date = kwargs.pop('thread_start_date', datetime.now())

        if 'key' not in kwargs and thread:
            thread_key = get_thread_suburban_key(thread.number, thread.path[0].station)
            sk, _ = SuburbanKey.objects.get_or_create(thread=thread, key=thread_key)
            thread_key = sk.key
            kwargs['key'] = ThreadKey(thread_key=thread_key, thread_start_date=thread_start_date)

        th_event = super(ThreadEventsFactory, cls).create(**kwargs)

        if thread:
            th_event.thread = thread
            th_event.rts_path = list(thread.path)

        return th_event


class ThreadStationStateFactory(MongoEngineFactory):
    class Meta:
        model = ThreadStationState
    passed_several_times = False

    @classmethod
    def create_from_rtstation(cls, rts, thread_start_date, arrival=None, departure=None, passed_several_times=False):
        thread = rts.thread

        if arrival == {}:
            arrival_state = None
        else:
            default_arrival = {
                'dt': datetime(2017, 3, 20, 17, 11),
                'type': u'fact',
                'thread_uid': thread.uid,
                'minutes_from': 1,
                'minutes_to': 1
            }
            default_arrival.update(arrival or {})
            arrival_state = EventStateFactory(**default_arrival)

        if departure == {}:
            departure_state = None
        else:
            default_departure = {
                'dt': datetime(2017, 3, 20, 17, 27),
                'type': u'fact',
                'thread_uid': thread.uid,
                'minutes_from': 7,
                'minutes_to': 7
            }
            default_departure.update(departure or {})
            departure_state = EventStateFactory(**default_departure)

        thread_key = get_thread_suburban_key(thread.number, thread.path[0].station)
        sub_key_1, created = SuburbanKey.objects.get_or_create(thread=thread, key=thread_key)

        tss = ThreadStationStateFactory(
            key=ThreadStationKeyFactory(
                thread_key=sub_key_1.key,
                thread_start_date=thread_start_date,
                station_key=get_rtstation_key(rts),
                arrival=rts.tz_arrival,
                departure=rts.tz_departure
            ),
            arrival_state=arrival_state,
            departure_state=departure_state,
            arrival=rts.tz_arrival,
            departure=rts.tz_departure,
            tz=rts.station.time_zone,
            passed_several_times=passed_several_times
        )
        tss.rts = rts
        return tss


class ThreadStationKeyFactory(MongoEngineFactory):
    class Meta:
        model = ThreadStationKey


class EventStateFactory(MongoEngineFactory):
    class Meta:
        model = EventState


class StationUpdateInfoFactory(MongoEngineFactory):
    class Meta:
        model = StationUpdateInfo


class StationEventFactory(MongoEngineFactory):
    class Meta:
        model = StationEvent

    time = 0
    twin_key = ''
    weight = 0.1
    passed_several_times = False
    dt_normative = datetime.now()


class StationExpectedEventFactory(MongoEngineFactory):
    class Meta:
        model = StationExpectedEvent


class HourEventsRateFactory(MongoEngineFactory):
    class Meta:
        model = HourEventsRate


class CompanyCrashFactory(MongoEngineFactory):
    class Meta:
        model = CompanyCrash


class MovistaCancelRawFactory(MongoEngineFactory):
    class Meta:
        model = MovistaCancelRaw


class StationCancelFactory(MongoEngineFactory):
    class Meta:
        model = StationCancel

    @classmethod
    def create_from_rtstations(cls, rtstations, **kwargs):
        kwargs['cancelled_stations'] = list(map(CancelledStation.create_from_rts, rtstations))
        return super(StationCancelFactory, cls).create(**kwargs)
