# coding: utf8

import logging
from collections import defaultdict
from functools import partial
from itertools import chain

from common.apps.suburban_events import models
from common.apps.suburban_events.utils import ThreadKey, get_threads_suburban_keys, get_thread_type_and_clock_dir
from common.db.mongo.bulk_buffer import BulkBuffer
from travel.rasp.library.python.common23.date.environment import now
from travel.rasp.library.python.common23.logging import log_run_time

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


class Event(object):
    """ Интерфейс сматченного события, позволяющий правильно его сохранить. """

    @property
    def rtstation(self):
        raise NotImplementedError

    @property
    def thread(self):
        raise NotImplementedError

    @property
    def thread_start_date(self):
        raise NotImplementedError

    @property
    def dt_normative(self):
        raise NotImplementedError

    @property
    def dt_fact(self):
        raise NotImplementedError

    @property
    def twin_key(self):
        """ Ключ для разделения двух одинаковых станций в одной нитке."""
        raise NotImplementedError

    @property
    def passed_several_times(self):
        """ Проходит ли нитка такую станцию более чем 1 раз."""
        raise NotImplementedError

    @property
    def type(self):
        """ Тип события - arrival/departure."""
        raise NotImplementedError

    @property
    def weight(self):
        raise NotImplementedError


def prepare_events(events):
    threads = [e.thread for e in events]
    suburban_keys = get_threads_suburban_keys(threads)
    by_thread_key = defaultdict(list)
    last_time_by_station = {}

    for event in events:
        try:
            thread_type, clock_dir = get_thread_type_and_clock_dir(event.thread)
            thread_key = ThreadKey(suburban_keys[event.thread.id], event.thread_start_date, thread_type, clock_dir)
            by_thread_key[thread_key].append(event)
        except Exception as ex:
            log.exception('thread {} failed: {}'.format(event.thread.uid, repr(ex)))

        last_time = last_time_by_station.get(event.rtstation.station.id)
        if not last_time or (event.dt_fact > last_time):
            last_time_by_station[event.rtstation.station.id] = event.dt_fact

    save_stations_info(last_time_by_station)

    return by_thread_key


def prepare_mczk_events(events):
    by_thread_key = defaultdict(list)
    last_time_by_station = {}

    for event in events:
        try:
            key = event.th_event.key
            thread_key = ThreadKey(key.thread_key, key.thread_start_date, key.thread_type, key.clock_direction)
            by_thread_key[thread_key].append(event)
        except Exception as ex:
            log.exception('th_events {} failed: {}'.format(key, repr(ex)))

        last_time = last_time_by_station.get(event.station_key)
        if not last_time or (event.dt_fact > last_time):
            last_time_by_station[event.station_key] = event.dt_fact

    save_stations_info(last_time_by_station)

    return by_thread_key


def save_stations_info(last_time_by_station):
    with BulkBuffer(models.StationUpdateInfo._get_collection(), max_buffer_size=2000, logger=log) as coll:
        for station_key, timestamp in last_time_by_station.items():
            coll.update_one(
                {
                    'station_key': station_key
                },
                {
                    '$set': {'timestamp': timestamp}
                },
                upsert=True
            )


def save_prepared_events(by_thread_key):
    if not by_thread_key:
        return

    dt_save = now()
    th_objs = models.ThreadEvents.objects.filter(
        __raw__={
            '$or': [{
                'key.thread_key': th_key.thread_key,
                'key.thread_start_date': th_key.thread_start_date,
                'key.thread_type': th_key.thread_type,
                'key.clock_direction': th_key.clock_direction
            } for th_key in by_thread_key.keys()]
        }
    )
    th_events_key = {
        ThreadKey(
            th_obj.key.thread_key,
            th_obj.key.thread_start_date,
            th_obj.key.thread_type,
            th_obj.key.clock_direction): th_obj
        for th_obj in th_objs
    }

    with log_run_time('save {} stations events'.format(len(list(chain(*by_thread_key.values()))))):
        with BulkBuffer(models.ThreadEvents._get_collection(), max_buffer_size=2000, logger=log) as coll:
            for thread_key, th_station_events in by_thread_key.items():
                last_event = sorted(th_station_events, key=lambda e: e.dt_normative, reverse=True)[0]
                # Обновляем последнее полученное событие нитки.
                try:
                    obj_last = th_events_key[thread_key].last_station_event
                except Exception as ex:
                    obj_last = None
                    log.exception('thread_key {} failed: {}'.format(thread_key, repr(ex)))

                if obj_last and obj_last.dt_normative > last_event.dt_normative:
                    last_station_event = obj_last
                else:
                    last_station_event = models.StationEvent(
                        station_key=last_event.station_key,
                        type=last_event.type,
                        dt_normative=last_event.dt_normative,
                        dt_fact=last_event.dt_fact,
                        time=last_event.time,
                        weight=last_event.weight,
                        twin_key=last_event.twin_key,
                        passed_several_times=last_event.passed_several_times)

                coll.update_one(
                    {
                        'key.thread_key': thread_key.thread_key,
                        'key.thread_start_date': thread_key.thread_start_date,
                        'key.thread_type': thread_key.thread_type,
                        'key.clock_direction': thread_key.clock_direction
                    },
                    {
                        '$set': {
                            'last_station_event': last_station_event.to_mongo(),
                            'need_recalc': True,
                        },
                        '$push': {'stations_events': {'$each': [
                            models.StationEvent(
                                station_key=e.station_key,
                                type=e.type,
                                dt_normative=e.dt_normative,
                                dt_fact=e.dt_fact,
                                time=e.time,
                                weight=e.weight,
                                twin_key=e.twin_key,
                                passed_several_times=e.passed_several_times,
                                dt_save=dt_save
                            ).to_mongo() for e in th_station_events
                        ]}},
                    },
                    upsert=True,
                )
