# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from common.apps.suburban_events.models import EventState
from common.apps.suburban_events.utils import EventStateType
from common.dynamic_settings.default import conf

MIN_LATES_DIFF = 1


def get_interpolated_fact_state(minutes_from, minutes_to, rts):
    return EventState(
        type=EventStateType.FACT_INTERPOLATED,
        minutes_from=minutes_from,
        minutes_to=minutes_to,
        thread_uid=rts.thread.uid,
    )


def interpolate_tss_states(prev_fact, next_fact, tss):
    """
    Интерполируем minutes_from и minutes_to на tss,
    используя имеющиеся факты от соседних станций - prev_fact и next_fact

    https://st.yandex-team.ru/RASPEXPORT-363
    """

    # сейчас у нас всегда minutes_from == minutes_to для FACT.
    prev_late, next_late = prev_fact.minutes_from, next_fact.minutes_from

    # считаем опережение как отсутствие опоздания
    prev_late = max(prev_late, 0)
    next_late = max(next_late, 0)

    if prev_late == next_late == 0:  # всё по расписанию
        minutes_from, minutes_to = 0, 0
    else:
        max_late = max(prev_late, next_late)
        if prev_late == 0 or next_late == 0:  # если только с одной стороны опоздание, то берем интервал [1, max]
            minutes_from, minutes_to = 1, max_late
        elif abs(prev_late - next_late) <= MIN_LATES_DIFF:  # при малом отличии опозданий, берем максимальное
            minutes_from, minutes_to = max_late, max_late
        else:
            min_late = min(prev_late, next_late)
            minutes_from, minutes_to = min_late, max_late

    if not tss.arrival_state:
        tss.set_arrival_state(get_interpolated_fact_state(minutes_from, minutes_to, tss.rts))

    if not tss.departure_state:
        tss.set_departure_state(get_interpolated_fact_state(minutes_from, minutes_to, tss.rts))


def interpolate_tss_path(tss_path):
    """
    tss_path - список tss, часть из которых содержит факты, часть - нет.
    Пытаемся интерполировать недостающие факты.

    https://st.yandex-team.ru/RASPEXPORT-363
    """

    if conf.SUBURBAN_ENABLE_FACT_INTERPOLATION:
        prev_fact = None
        need_interpolate = []
        for tss in tss_path:
            if not tss.arrival_state and not tss.departure_state:
                need_interpolate.append(tss)
                continue

            if not tss.arrival_state:
                next_fact = tss.departure_state
                need_interpolate.append(tss)  # есть факт отправления, но прибытия нет - надо его интерполировать
            else:
                next_fact = tss.arrival_state

            if prev_fact:
                for interp_tss in need_interpolate:
                    interpolate_tss_states(prev_fact, next_fact, interp_tss)
            need_interpolate = []

            if not tss.departure_state:
                prev_fact = tss.arrival_state
                need_interpolate.append(tss)  # есть факт прибытия, но отправления нет - надо его интерполировать
            else:
                prev_fact = tss.departure_state

    return [tss for tss in tss_path if tss.arrival_state or tss.departure_state]
