# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict
from datetime import datetime, timedelta
from itertools import groupby, chain
from typing import Callable, Dict, Iterable, List, Tuple

from django.db.models import Q

from common.apps.suburban_events import models, dynamic_params
from common.apps.suburban_events.forecast.events import log_run_time
from common.apps.suburban_events.forecast.fact_interpolation import interpolate_tss_path
from common.apps.suburban_events.logs import log_suburban_events
from common.apps.suburban_events.models import ThreadStationState, ThreadEvents, ForecastRepresentation
from common.apps.suburban_events.utils import (
    get_rtstation_key, MAX_DIFF_FROM_EVENT, get_threads_by_suburban_keys, EventStateType, get_thread_suburban_key,
    get_suburban_key_number_station, ThreadEventsTypeCodes, get_thread_type_and_clock_dir,
    prefix_for_dict_keys, EventStateSubType, collect_rts_by_threads
)
from common.db.mongo.bulk_buffer import BulkBuffer
from common.dynamic_settings.default import conf
from common.models.geo import Station
from common.models.schedule import RTStation, TrainTurnover
from common.utils.date import MSK_TZ, RunMask
from travel.rasp.library.python.common23.date.environment import now
from common.utils.multiproc import run_parallel, get_cpu_count

log = logging.getLogger(__name__)


def get_thread_station_state(th_event, rts):
    # type: (ThreadEvents, RTStation) -> models.ThreadStationState

    tss = models.ThreadStationState(
        key=models.ThreadStationKey(
            thread_key=th_event.suburban_key,
            thread_start_date=th_event.start_dt,
            station_key=get_rtstation_key(rts),
            thread_type=th_event.key.thread_type,
            clock_direction=th_event.key.clock_direction,
            arrival=rts.tz_arrival,
            departure=rts.tz_departure
        ),
        arrival=rts.tz_arrival,
        departure=rts.tz_departure,
        tz=rts.station.time_zone,  # на момент 2017.02.01 rts.pytz берет таймзону из станции, используем её
    )

    tss.rts = rts
    tss.th_event = th_event

    return tss


def event_to_fact_state(event, rts, rts_event, thread_start_dt):
    dt_fact_local = MSK_TZ.localize(event.dt_fact).astimezone(rts.station.pytz).replace(tzinfo=None)
    rts_event_local = rts.get_event_loc_dt(rts_event, thread_start_dt).replace(tzinfo=None)

    # Если отклонение от графка внутри этих границ, то считаем, что отклонения нет
    if -conf.SUBURBAN_MIN_EXTRA_TIME_FOR_FACT_DT < event.minutes_diff < conf.SUBURBAN_MIN_DELAY_FOR_FACT_DT:
        event_state = models.EventState(
            type=EventStateType.FACT,
            dt=rts_event_local,
            minutes_from=0,
            minutes_to=0,
            thread_uid=rts.thread.uid,
        )
    else:
        event_state = models.EventState(
            type=EventStateType.FACT,
            dt=dt_fact_local,
            minutes_from=event.minutes_diff,
            minutes_to=event.minutes_diff,
            thread_uid=rts.thread.uid,
        )

    event_state.original_event = event

    return event_state


def get_fact_state(th_event, rts, arrival_event, departure_event):
    ts_state = get_thread_station_state(th_event, rts)

    passed_several_times = False
    if arrival_event:
        ts_state.set_arrival_state(event_to_fact_state(arrival_event, rts, 'arrival', th_event.start_dt))
        passed_several_times |= arrival_event.passed_several_times

    if departure_event:
        ts_state.set_departure_state(event_to_fact_state(departure_event, rts, 'departure', th_event.start_dt))
        passed_several_times |= departure_event.passed_several_times

    ts_state.passed_several_times = passed_several_times

    return ts_state


def get_possible_delay_event_state(th_event, rts, delay, forecast_dt, deep, sub_type):
    representation = Forecaster.get_representation(sub_type, int(round(delay)), int(round(deep)))

    event_state = models.EventState(
        type=EventStateType.POSSIBLE_DELAY,
        thread_uid=rts.thread.uid,
        sub_type=sub_type,
        forecast_dt=MSK_TZ.localize(forecast_dt).astimezone(rts.pytz).replace(tzinfo=None),
        forecast_delay=delay,
        time_from_delay_station=deep,
        trigger=th_event.trigger_station_event.station_key,
        minutes_from=representation.minutes_from if representation else None,
        minutes_to=representation.minutes_to if representation else None
    )

    return event_state


def get_fact_possible_delay_state(th_event, rts, arr_dt, dep_dt, arr_delay, dep_delay, arr_deep, dep_deep,
                                  skip_arrival=False):
    """
    Функция генерации состояний опозданий для событий типа fact.
    :return: ts_state
    """
    if (arr_delay and arr_delay > conf.SUBURBAN_EVENT_PROPAGATION_DELTA) or \
            (dep_delay and dep_delay > conf.SUBURBAN_EVENT_PROPAGATION_DELTA):
        ts_state = get_thread_station_state(th_event, rts)

        if not skip_arrival and rts.tz_arrival and arr_delay >= conf.SUBURBAN_EVENT_PROPAGATION_DELTA:
            arrival_state = get_possible_delay_event_state(
                th_event, rts, arr_delay, arr_dt, arr_deep, EventStateSubType.FACT
            )
            ts_state.set_arrival_state(arrival_state)

        if rts.tz_departure and dep_delay >= conf.SUBURBAN_EVENT_PROPAGATION_DELTA:
            departure_state = get_possible_delay_event_state(
                th_event, rts, dep_delay, dep_dt, dep_deep, EventStateSubType.FACT
            )
            ts_state.set_departure_state(departure_state)

        return ts_state


def filter_events_by_weights(stations_events):
    """ Среди всех событий одного типа для каждого rts оставляем только события с наибольшим весом. """

    last_stations_events = defaultdict(list)
    filtered_events = []

    for event in stations_events:
        last_stations_events[(event.rts, event.type)].append(event)

    for rts_events in last_stations_events.values():
        max_weight = max(e.weight for e in rts_events)
        filtered_events.extend([event for event in rts_events if event.weight == max_weight])

    return filtered_events


def find_rts_for_events(th_event):
    """
    Для каждого события находим ближайший rts.
    Сохраняем порядковый номер rts в пути нитки в поле station_idx.
    Сохраняем ближайший rts в поле rts.
    Если подходящего rts не нашлось, то событие пропускается.
    """
    rtss_by_station = defaultdict(list)
    idx_by_rts = dict()
    filtered_events = []

    for i, rts in enumerate(th_event.rts_path):
        rtss_by_station[rts.station.id].append(rts)
        idx_by_rts[rts] = i

    for event in th_event.stations_events:
        station_rtss = rtss_by_station[int(event.station_key)]
        rts = get_closest_rts(event, station_rtss, th_event.start_dt)
        if rts:
            event.station_idx = idx_by_rts[rts]
            event.rts = rts
            filtered_events.append(event)

    return filtered_events


def get_last_rts_events(stations_events):
    """
    Возвращаем словарь, который по rts и типу события (arrival/departure) возвращает последнее событие.
    Возвращаем событие, относящееся к наибольшему rts в порядке следования в нитке.
    """
    last_stations_events = defaultdict(list)
    last_rts_events = dict()
    last_events = []

    for event in stations_events:
        last_stations_events[(event.rts, event.type)].append(event)

    for (rts, event_type), rts_events in last_stations_events.items():
        last_event = sorted(rts_events, key=lambda e: e.dt_save)[-1]
        last_rts_events[(rts, event_type)] = last_event
        last_events.append(last_event)

    last_event = sorted(last_events, key=lambda e: e.rts_dt_normative, reverse=True)[0]

    return last_rts_events, last_event


def get_cancelled_states(cancels):
    # type: (List[ThreadStationState]) -> Dict[Tuple[RTStation, unicode], models.EventState]

    cancels_by_rts = {}

    for cancel in cancels:
        rts = cancel.rts

        arrival_cancel = cancel.arrival_state
        if arrival_cancel:
            cancels_by_rts[(rts, 'arrival')] = arrival_cancel

        departure_cancel = cancel.departure_state
        if departure_cancel:
            cancels_by_rts[(rts, 'departure')] = departure_cancel

    return cancels_by_rts


def get_closest_rts(event, rtstations, thread_start_dt):
    """
    Находим ближайший к событию rts.
    Сохраняем разницу между фактическим и нормативным временем в поле minutes_diff.
    Сохраняем нормативное время в поле rts_dt_normative.
    """
    diffs = []
    for rts in rtstations:
        rts_time = getattr(rts, 'tz_{}'.format(event.type))
        if rts_time is not None:
            rts_dt_normative = rts.get_event_dt(event.type, thread_start_dt, out_tz=MSK_TZ).replace(tzinfo=None)
            diff = (event.dt_fact - rts_dt_normative).total_seconds() / 60
            if -conf.SUBURBAN_MAX_EVENT_OVERTAKING <= diff <= conf.SUBURBAN_MAX_EVENT_DELAY:
                diffs.append((abs(event.time - rts_time), rts, diff, rts_dt_normative))

    if diffs:
        _, rts, diff, rts_dt_normative = sorted(diffs, key=lambda x: x[0])[0]

        if event.passed_several_times and diff > MAX_DIFF_FROM_EVENT:
            return

        event.minutes_diff = diff
        event.rts_dt_normative = rts_dt_normative
        return rts


def filter_events_by_prev_max_next_min(stations_events):
    """
    https://st.yandex-team.ru/RASPEXPORT-298
    Проводим фильтрацию событий обычных ниток.
    Обрабатываем события в порядке их поступления от РЖД.
    Внутри одного пакета РЖД события обрабатываются в порядке следования станций в нитке.
    Для каждого события проверяем, больше ли его время максимального времени
    среди уже валидных событий предыдущих (в порядке следования) станций.
    Проверяем, меньше ли время события минимального времени последующих станций.
    Если событие не удовлетворяет условиям, то оно отбрасывается.
    """
    filtered_events = []

    for dt_save, pack_events in groupby(sorted(stations_events, key=lambda x: x.dt_save), key=lambda x: x.dt_save):
        pack_events = list(pack_events)
        pack_events.sort(key=lambda x: x.station_idx)
        for pack_event in pack_events:
            max_before, min_after = None, None
            for filtered_event in filtered_events:
                if filtered_event.station_idx < pack_event.station_idx:
                    max_before = max(max_before, filtered_event.dt_fact) if max_before else filtered_event.dt_fact
                elif filtered_event.station_idx > pack_event.station_idx:
                    min_after = min(min_after, filtered_event.dt_fact) if min_after else filtered_event.dt_fact

            if (max_before and max_before >= pack_event.dt_fact) or (min_after and min_after <= pack_event.dt_fact):
                continue
            filtered_events.append(pack_event)

    return filtered_events


def get_fact_states_unknown_rts(th_event, cancels):
    """
    Функция возвращает фактические события для rts, по которым пришли данные.
    :param th_event: События рассматриваемой нитки.
    :return:
    thread_station_states список событий по фактическим rts,
    unknown_state_rts список оставшихся rts до конца маршрута,
    last_event последнее известное событие,
    arrival_event событие прибытия на последний известный rts,
    rts последний rts,
    total_stop_minutes общее количество минут, затраченное на остановки.
    """
    events_with_rts = find_rts_for_events(th_event)
    filtered_events = filter_events_by_weights(events_with_rts)

    if th_event.key.thread_type != ThreadEventsTypeCodes.MCZK:
        # https://st.yandex-team.ru/RASPEXPORT-298
        # Проводим фильтрацию событий обычных ниток.
        try:
            filtered_events = filter_events_by_prev_max_next_min(filtered_events)
        except Exception as ex:
            log.exception('filter_events_by_prev_max_next_min failed: {}'.format(repr(ex)))

    if not filtered_events:
        return None, None, None

    last_rts_events, last_event = get_last_rts_events(filtered_events)
    cancelled_states_by_rts = get_cancelled_states(cancels)

    tss_path = []
    unknown_state_rts = []
    total_stop_minutes = 0

    # Добавляем фактические события
    for i, rts in enumerate(th_event.rts_path):
        if rts.tz_departure and rts.tz_arrival:
            total_stop_minutes += calc_stop_diff_time(rts.tz_departure - rts.tz_arrival)

        arrival_event = last_rts_events.get((rts, 'arrival'))
        departure_event = last_rts_events.get((rts, 'departure'))

        arrival_cancel = cancelled_states_by_rts.get((rts, 'arrival'))
        departure_cancel = cancelled_states_by_rts.get((rts, 'departure'))

        if arrival_event or departure_event:
            tss = get_fact_state(
                th_event, rts,
                None if arrival_cancel else arrival_event,
                None if departure_cancel else departure_event
            )

            tss_path.append(tss)

            # Дошли до последнего известного события - выходим и переходим к прогнозу
            if last_event in [arrival_event, departure_event]:
                # Все станции после текущей точно не имеют событий
                unknown_state_rts = th_event.rts_path[i + 1:]
                break
        else:
            tss_path.append(get_thread_station_state(th_event, rts))

    thread_station_states = interpolate_tss_path(tss_path)

    th_event.last_event = last_event
    th_event.last_event_is_arrival = last_event == arrival_event

    return thread_station_states, unknown_state_rts, total_stop_minutes


def get_possible_ok_states(th_event, ts_states):
    """
    Ищем rts с хвоста, для которых нет фактов/прогнозов опозданий, и выставляем для них "ожидается по расписанию"
    """

    by_rts = {tss.rts: tss for tss in ts_states}
    possible_ok_rts = []
    for rts in reversed(th_event.rts_path):
        tss = by_rts.get(rts)
        if tss:
            # нашли первый tss с прогнозами, выходим
            if tss.departure_state:
                break

            # нашли первый tss с прогнозами, но departure у него не расчитан; добавляем tss к расчету и выходим
            if tss.arrival_state:
                possible_ok_rts.append(rts)
                break
        else:
            # rts без прогнозов
            possible_ok_rts.append(rts)

    ts_states = set()
    for rts in possible_ok_rts:
        tss = by_rts.get(rts)
        if not tss:
            tss = get_thread_station_state(th_event, rts)

        if rts.tz_arrival and not tss.arrival_state:
            tss.set_arrival_state(models.EventState(
                type=EventStateType.POSSIBLE_OK,
                thread_uid=rts.thread.uid,
            ))
            ts_states.add(tss)

        if rts.tz_departure and not tss.departure_state:
            tss.set_departure_state(models.EventState(
                type=EventStateType.POSSIBLE_OK,
                thread_uid=rts.thread.uid,
            ))
            ts_states.add(tss)

    return ts_states


def get_fact_unknown_states(th_event, unknown_state_rts, thread_station_states):
    last_event = th_event.last_event
    th_event.trigger_station_event = last_event

    # Если событие по безостановочной станции, то в качестве нормативного времени используем время РЖД.
    # https://st.yandex-team.ru/RASPEXPORT-291
    if last_event.rts.is_no_stop():
        delay = (last_event.dt_fact - last_event.dt_normative).total_seconds() / 60
    else:
        delay = last_event.minutes_diff

    if check_forecast_exists(delay, conf.SUBURBAN_MIN_DELAY_TO_FORECAST):
        stop_minutes = 0
        # Для текущей станции может быть неизвестно отправление - считаем прогноз для него сразу
        if th_event.last_event_is_arrival:
            if last_event.rts.tz_departure and last_event.rts.tz_arrival:
                minutes_diff = last_event.rts.tz_departure - last_event.rts.tz_arrival
                stop_minutes += calc_stop_diff_time(minutes_diff)

            rts_departure_msk_dt = last_event.rts.get_event_dt('departure', th_event.start_dt, out_tz=MSK_TZ)
            if rts_departure_msk_dt:
                rts_departure_msk_dt = rts_departure_msk_dt.replace(tzinfo=None)
                dep_delay, dep_dt, dep_deep = calc_rts_event_delay_params(
                    th_event, delay, rts_departure_msk_dt, stop_minutes
                )
                forecast_ts_state = get_fact_possible_delay_state(
                    th_event, last_event.rts, None, dep_dt, None, dep_delay, None, dep_deep, skip_arrival=True)
                if forecast_ts_state:
                    thread_station_states[-1].departure_state = forecast_ts_state.departure_state

        unknown_rts_states = get_unknown_rtstations_possible_states(
            th_event, unknown_state_rts, stop_minutes, delay, get_fact_possible_delay_state,
            conf.SUBURBAN_MIN_DELAY_TO_FORECAST
        )

        return unknown_rts_states


def get_base_events_states(th_event, cancels):
    if not th_event.stations_events:
        return []

    thread_station_states, unknown_state_rts, _ = get_fact_states_unknown_rts(th_event, cancels)
    if not thread_station_states:
        return []

    unknown_rts_states = get_fact_unknown_states(th_event, unknown_state_rts, thread_station_states)
    if unknown_rts_states:
        thread_station_states.extend(unknown_rts_states)

    return thread_station_states


def check_forecast_exists(trigger_delay, delay_const):
    return trigger_delay > delay_const


class Forecaster(object):
    # Используется для расшаривания текущего интсанса forecaster'а между воркерами в multiprocessing
    instance = None

    def __init__(self):
        Forecaster.instance = self
        self.forecast_uid = datetime.utcnow().isoformat()

        self.msk_now = now()
        self.query_date = self.msk_now.replace(hour=0, minute=0, second=0, microsecond=0)

        self.last_query_time = dynamic_params.get_param('last_successful_query_time')
        if not self.last_query_time:
            self.last_query_time = self.msk_now

        # Итоговые прогнозы.
        # thread_station_states[th_events.key] = [event_state1, event_state2, ...]
        self.thread_station_states = defaultdict(list)

        # ThreadEvents, для которых пересчет больше не нужен.
        # Для них устанавливается need_recalc=False в конце работы.
        self.thread_events_no_need_recalc = []

        # Нитки с отменами
        self.threads_with_cancels = {}
        self.fully_cancelled_threads = set()
        self.threads_with_cancels_to_save = set()

        # Данные про нитки для просчета оборотов (turnover). Собирается до запуска add_turnover_forecast.
        self.threads_for_turnover = {}

        # Данные про нитки для вычисления событий possible_ok. Собирается до запуска add_possible_ok_forecast.
        self.threads_for_possible_ok = {}

        self.crash_dt_by_company = get_companies_crashes()
        self.forecast_representations = get_forecast_representations()
        self._active_forecast_stations = None

    def run(self):
        self._safe_method_call(self.add_cancels_forecast, conf.SUBURBAN_ENABLE_CANCELS)

        with log_run_time('add_base_forecast'):
            self.add_base_forecast()

        self._safe_method_call(self.add_no_data_forecast, conf.SUBURBAN_ENABLE_NO_DATA)
        self._safe_method_call(self.add_no_next_data_forecast, conf.SUBURBAN_ENABLE_NO_NEXT_DATA)
        self._safe_method_call(self.add_turnover_forecast, conf.SUBURBAN_TRAIN_TURNOVER_ENABLED)
        self._safe_method_call(self.add_possible_ok_forecast, conf.SUBURBAN_ENABLE_POSSIBLE_OK)
        self._safe_method_call(self.add_mczk_forecast, conf.SUBURBAN_ENABLE_MCZK)

        self.merge_cancels()
        self.save_forecast()
        self.save_thread_events_no_need_recalc()

    def _safe_method_call(self, method, check=True):
        # type: (Callable, bool) -> None
        if not check:
            return

        name = method.__func__.__name__
        with log_run_time(name):
            try:
                method()
            except Exception as ex:
                log.exception('{} failed: {}'.format(name, repr(ex)))

    def add_base_forecast(self):
        with log_run_time('get thread events'):
            thread_events = list(models.ThreadEvents.objects.filter(
                key__thread_type__ne=ThreadEventsTypeCodes.MCZK,
                need_recalc=True,
            ))

        with log_run_time('get_events_states'):
            for th_event in self.setup_thread_events(thread_events):
                cancels = self.threads_with_cancels.get(th_event.key, [])
                if cancels:
                    self.threads_with_cancels_to_save.add(th_event.key)
                events_states = get_base_events_states(th_event, cancels)
                if events_states:
                    self.thread_station_states[th_event.key].extend(events_states)
                    self.threads_for_turnover[th_event.key] = th_event
                    self.threads_for_possible_ok[th_event.key] = th_event

        self.thread_events_no_need_recalc.extend(thread_events)

    def get_expected_events_conditions(self):
        """
        https://st.yandex-team.ru/RASPEXPORT-323
        Просматриваем перевозчиков с авариями.
        Если известно только время начала аварии, то при построении прогноза могут быть использованы только станции
        нитки, которые должны были быть пройдены до аварии.
        Если известно время окончания аварии, то при построении прогноза также могут быть использованы станции
        нитки, которые должны были быть пройдены после аварии.
        Для ниток, которые не содержат перевозчика или по перевозчику нет аварии логика не меняется.
        """
        actual_stations = [str(st.id) for st in self.active_forecast_stations if st.use_in_departure_forecast]
        not_gone_delay_time = self.last_query_time - timedelta(minutes=conf.SUBURBAN_MIN_DELAY_FOR_NO_DATA_EVENT)

        crashed_companies = []
        expected_events_conditions = []

        for company, crash in self.crash_dt_by_company.items():
            crashed_companies.append(company)
            normative_conditions = [{'dt_normative': {'$lt': min(not_gone_delay_time, crash.first_dt)}}]
            if crash.last_dt:
                normative_conditions.append({'dt_normative': {'$gt': crash.last_dt, '$lte': not_gone_delay_time}})

            expected_events_conditions.append({
                'stations_expected_events': {'$elemMatch': {
                    '$or': normative_conditions,
                    'station_key': {'$in': actual_stations},
                }},
                'thread_company': company
            })

        expected_events_conditions.append({
            'stations_expected_events': {'$elemMatch': {
                'dt_normative': {'$lte': not_gone_delay_time},
                'station_key': {'$in': actual_stations},
            }},
            'thread_company': {'$nin': crashed_companies},
        })

        return expected_events_conditions

    def add_no_data_forecast(self):
        """
        Функция достает из монги нитки для которых не было получено событий, но они уже должны были пройти
        хотя бы одну станцию из тех, по которым приходят данные согласно нормативному времени.
        Исключаются нитки, которые на текущий момент уже должны были прибыть.
        """
        expected_events_conditions = self.get_expected_events_conditions()

        run_thread_empty_events = list(models.ThreadEvents.objects(
            __raw__={
                'key.thread_start_date': {'$gte': self.query_date, '$lt': self.query_date + timedelta(days=1)},
                'key.thread_key': {'$regex': '^[67]'},
                '$and': [
                    {'$or': [{'stations_events': {'$exists': False}}, {'stations_events': {'$size': 0}}]},
                    {'$or': expected_events_conditions}
                ],
                '$where': """this.stations_expected_events[this.stations_expected_events.length - 1].dt_normative >
                ISODate('{}')""".format(self.msk_now)
            }
        ))

        for th_event in self.setup_thread_events(run_thread_empty_events):
            cancels = self.threads_with_cancels.get(th_event.key, [])
            if cancels:
                self.threads_with_cancels_to_save.add(th_event.key)
            no_data_rtstations = get_rtstations_for_no_data(th_event, cancels, self.msk_now)
            if no_data_rtstations:
                events_states = get_no_data_events_states(th_event, no_data_rtstations)
                self.thread_station_states[th_event.key].extend(events_states)

    def add_no_next_data_forecast(self):
        """
        Функция достает из монги нитки для которых в прошлом было получено хотябы одно событие,
        но вовремя не были получены последующие ожидаемые события.
        Ожидаемые события - события нитки, для станций которых приходят события по другим ниткам.
        """
        actual_stations = [str(st.id) for st in self.active_forecast_stations if st.use_in_forecast]
        not_gone_delay_time = self.last_query_time - timedelta(minutes=conf.SUBURBAN_MIN_DELAY_FOR_NO_NEXT_DATA_EVENT)

        only_arrival = ''
        if not conf.SUBURBAN_NO_NEXT_DATA_ONLY_ARRIVAL:
            only_arrival = '&& (obj.last_station_event.station_key != o.station_key)'

        map_function = """
         function get_station(obj) {
             var actual_stations = %s;
             var not_gone_delay = ISODate('%s');

             var dep_time = obj.last_station_event.dt_fact.getTime();
             if (obj.last_station_event.type == 'arrival') {
                 if (obj.last_station_event.dt_fact >= obj.last_station_event.dt_normative) {
                     dep_time += 60000;
                 } else {
                     dep_time = obj.last_station_event.dt_normative.getTime();
                 }
             }

            for (var i in obj.stations_expected_events.slice(0, -1)) {
                o = obj.stations_expected_events[i];
                diff = (o.time - obj.last_station_event.time) * 60000;
                if ((diff > 0)
                    && (not_gone_delay.getTime() >= (dep_time + diff))
                    && (actual_stations.indexOf(o.station_key) >= 0) %s) {
                        return o;
                    }
            }
            return null;
         }

         this.trigger_station_event = get_station(this);
         emit(this._id, this);
         """ % (actual_stations, not_gone_delay_time, only_arrival)

        reduce_function = 'function(key, values) {return values[0];}'

        objs = models.ThreadEvents.objects(
            __raw__={
                'stations_events': {'$exists': True, '$not': {'$size': 0}},
                'last_station_event': {'$exists': True},
                'stations_expected_events': {'$exists': True, '$not': {'$size': 0}},
                'key.thread_start_date': {'$gte': self.query_date, '$lt': self.query_date + timedelta(days=1)},
                'key.thread_key': {'$regex': '^[67]'},
            }
        )

        run_thread_events_objs = objs.map_reduce(map_function, reduce_function, 'suburban_forecast_no_next_data_tmp')
        run_thread_events = []
        for obj in run_thread_events_objs:
            if obj.value.get('trigger_station_event'):
                obj.value.pop('_id')
                run_thread_events.append(models.ThreadEvents(**obj.value))

        for th_event in self.setup_thread_events(run_thread_events):
            try:
                if th_event.thread.company:
                    # https://st.yandex-team.ru/RASPEXPORT-323
                    crash = self.crash_dt_by_company.get(th_event.thread.company.id, None)
                    if crash and (not crash.last_dt or crash.last_dt >= th_event.last_station_event.dt_fact):
                        continue

                # пытаемся получить события, которые могли быть построены в base_forecast
                base_forecast = self.thread_station_states.get(th_event.key)
                cancels = self.threads_with_cancels.get(th_event.key, [])
                if cancels:
                    self.threads_with_cancels_to_save.add(th_event.key)
                events_states = get_no_next_data_events_states(th_event, self.last_query_time, base_forecast, cancels)

                self.thread_station_states[th_event.key] = events_states
                self.threads_for_turnover[th_event.key] = th_event
                self.threads_for_possible_ok[th_event.key] = th_event
            except Exception as ex:
                log.exception('no_next_data th_event {} failed: {}'.format(th_event.key, repr(ex)))
                continue

    def add_turnover_forecast(self):
        """
        https://st.yandex-team.ru/RASPEXPORT-259
        Функция использует события опозданий, полученных по фактическим данным или по "застрявшим в пути поездам"
        для прогнозирования опозданий после "оборота поездов".
        Если событие опоздания распространено до последний станции нитки и в базе есть оборот для номера поезда и
        последней станции это нитки, то распространяем опоздания и на нитку, которая будет после оборота.

        Для правильной работы требуется заполнение self.threads_for_turnover и self.thread_station_states
        """
        today = now()
        turnover_delta = timedelta(minutes=conf.SUBURBAN_MIN_DELAY_FOR_TURNOVER)
        before_threads = []

        for th_event in self.threads_for_turnover.values():
            states = self.thread_station_states[th_event.key]
            if not states:
                continue

            try:
                last_expected_event = th_event.stations_expected_events[-1]
            except IndexError:
                log.error(u'Нет предпологаемых событий для нитки %s' % th_event)
                continue

            last_state = states[-1]
            arr_state = last_state.arrival_state
            if arr_state and arr_state.forecast_dt:
                station_equals = last_expected_event.station_key == last_state.key.station_key
                last_is_possible = arr_state.type == EventStateType.POSSIBLE_DELAY
                is_forecast_ge = arr_state.forecast_dt >= get_local_expected_event_time(last_expected_event,
                                                                                        last_state.rts)

                if station_equals and last_is_possible and is_forecast_ge:
                    before_threads.append((th_event.thread, last_expected_event.dt_normative,
                                           last_state.key.station_key, arr_state.forecast_dt, last_state.rts))

        numbers_after_date_station, forecast_before_by_after = get_threads_after_turnover_info(before_threads, today)

        filtered_after_thread_events, thread_before_by_th_event = find_thread_events_after_turnover(
            numbers_after_date_station, forecast_before_by_after, turnover_delta)

        for th_event in self.setup_thread_events(filtered_after_thread_events):
            trigger = thread_before_by_th_event[th_event].suburban_key.key
            events_states = get_turnover_events_states(th_event, trigger)
            self.thread_station_states[th_event.key].extend(events_states)

    def add_mczk_forecast(self):
        """ https://st.yandex-team.ru/RASPEXPORT-290 """
        with log_run_time('get thread events'):
            thread_events = list(models.ThreadEvents.objects.filter(
                key__thread_type=ThreadEventsTypeCodes.MCZK,
                need_recalc=True,
            ))

        with log_run_time('get_mczk_events_fact_states'):
            for th_event in self.setup_thread_events(thread_events):
                events_states, _, _ = get_fact_states_unknown_rts(th_event, [])
                if not events_states:
                    continue

                if events_states:
                    self.thread_station_states[th_event.key].extend(events_states)

        self.thread_events_no_need_recalc.extend(thread_events)

    def add_possible_ok_forecast(self):
        """
        Для всех подходящих ниток расчитываем прогноз "по расписанию"
        https://st.yandex-team.ru/SUBURBAN-23
        """
        for th_event_key, th_event in self.threads_for_possible_ok.items():
            tsses = self.thread_station_states[th_event_key]
            poss_ok_states = get_possible_ok_states(th_event, tsses)
            tsses.extend(poss_ok_states)

    def add_cancels_forecast(self):
        with log_run_time('get cancelled thread events'):
            thread_events = list(models.ThreadEvents.objects(
                __raw__={
                    'key.thread_start_date': {'$gte': self.query_date},
                    '$where': 'this.stations_cancels !== undefined && this.stations_cancels.length >= 1'
                }
            ))

        for th_event in self.setup_thread_events(thread_events):
            cancelled_thread_station_states, fully_cancelled = prepare_cancels(th_event)
            if not cancelled_thread_station_states:
                log.info('No prepared cancels for {}'.format(th_event.suburban_key))
                continue
            log.info('{} prepared cancels for {} (fully_cancelled: {}, need_recalc: {})'.format(
                len(cancelled_thread_station_states),
                th_event.suburban_key,
                fully_cancelled,
                th_event.need_recalc
            ))
            self.threads_with_cancels[th_event.key] = cancelled_thread_station_states
            if fully_cancelled:
                self.fully_cancelled_threads.add(th_event.key)
            if fully_cancelled or th_event.need_recalc:
                self.threads_with_cancels_to_save.add(th_event.key)

            self.thread_events_no_need_recalc.append(th_event)

    def merge_cancels(self):
        for th_key in self.threads_with_cancels_to_save:
            log.info('{} has cancels'.format(th_key.thread_key))
            cancelled_tsses = self.threads_with_cancels[th_key]
            base_tsses = self.thread_station_states.get(th_key, [])
            base_tsses_with_cancels = compare_tsses_cancels_first(base_tsses, cancelled_tsses)
            self.thread_station_states[th_key] = base_tsses_with_cancels

    @property
    def active_forecast_stations(self):
        if self._active_forecast_stations is None:
            with log_run_time('get active_forecast_stations'):
                # Получаем станции по которым приходили события в течении последних DAYS_FOR_ACTUAL_STATIONS дней.
                actual_stations = [str(station.station_key) for station in models.StationUpdateInfo.objects.filter(
                    timestamp__gte=(self.msk_now - timedelta(days=conf.SUBURBAN_DAYS_FOR_ACTUAL_STATIONS))
                )]

                self._active_forecast_stations = list(
                    Station.objects.
                    filter(id__in=actual_stations).
                    filter(Q(use_in_departure_forecast=True) | Q(use_in_forecast=True)).
                    only('use_in_departure_forecast', 'use_in_forecast')
                )

        return self._active_forecast_stations

    def setup_thread_events(self, thread_events):
        # type: (Forecaster, List[ThreadEvents]) -> List[ThreadEvents]

        """
        На каждой ThreadEvent сохраняем нитку и путь (rtstations).
        Убираем эвенты, для которых этих данных не нашлось.
        """
        with log_run_time('get threads for {} thread events'.format(len(thread_events))):
            thread_keys = set(te.suburban_key for te in thread_events)
            threads_by_keys = get_threads_by_suburban_keys(thread_keys)

        all_threads = set()
        with log_run_time('match events to threads'):
            for th_event in thread_events:
                if th_event.key in self.fully_cancelled_threads:
                    continue

                threads = threads_by_keys.get(th_event.suburban_key, [])

                found_threads = []
                for thread in threads:
                    if thread.runs_at(th_event.start_dt) and thread.tz_start_time == th_event.start_dt.time():
                        if th_event.key.thread_type == ThreadEventsTypeCodes.MCZK:
                            thread_type, clock_dir = get_thread_type_and_clock_dir(thread)
                            if not clock_dir == th_event.key.clock_direction:
                                continue

                        found_threads.append(thread)

                if not found_threads:
                    log.warning(u'Thread not found for {} {} ({})'.format(
                        th_event.suburban_key, th_event.start_dt, th_event.id))
                    continue

                if len(found_threads) > 1:
                    # Такой случай - ошибка контент-менеджеров. Пропускаем.
                    log.warning(u'Ambigous thread match for {} {} ({}) - {}'.format(
                        th_event.suburban_key, th_event.start_dt, th_event.id, [t.id for t in found_threads]))
                    continue

                th_event.thread = found_threads[0]
                all_threads.add(th_event.thread)

        with log_run_time('get rtstations for {} threads'.format(len(all_threads))):
            rtstations = list(
                RTStation.objects.filter(
                    thread_id__in=[t.id for t in all_threads]
                ).select_related(
                    'thread', 'station'
                ).only(
                    'tz_arrival', 'tz_departure', 'time_zone', 'departure_subdir',
                    'thread__id', 'thread__tz_start_time', 'thread__time_zone', 'thread__uid', 'thread__title',
                    'station__id', 'station__time_zone', 'station__settlement_id', 'station__title', 'thread__company',
                ).order_by('id')
            )

        with log_run_time('get pathes for threads of {} rtstaions'.format(len(rtstations))):
            pathes_by_thread = collect_rts_by_threads(rtstations)

            for th_event in thread_events:
                th_event.rts_path = pathes_by_thread[th_event.thread]

        return [th_event for th_event in thread_events if th_event.thread and th_event.rts_path]

    def save_forecast(self):
        pool_size = get_cpu_count()
        # На маленьких машинах пытаемся распараллелиться по максимуму,
        # а на больших - не занимать все ядра, чтобы не слишком пересекаться с другими процессами
        if pool_size >= 4:
            pool_size -= 2

        items_for_worker = int(len(self.thread_station_states) / pool_size) + 1

        workers_data = []
        for i in range(pool_size):
            workers_data.append((i * items_for_worker, (i + 1) * items_for_worker))

        total_opers_count = 0
        with log_run_time('updating tss: run in {} processes'.format(pool_size)):
            for opers_count in run_parallel(save_forecast_parallel, workers_data, pool_size):
                total_opers_count += opers_count
                log.info('worker done {} operations'.format(opers_count))

            log.info('total operations done: {}'.format(total_opers_count))

    def save_thread_events_no_need_recalc(self):
        with log_run_time('saving {} thread events'.format(len(self.thread_events_no_need_recalc))):
            with BulkBuffer(ThreadEvents._get_collection(), max_buffer_size=2000, logger=log) as coll:
                for thread_event in self.thread_events_no_need_recalc:
                    coll.update_many(
                        prefix_for_dict_keys(thread_event.key.to_mongo(), 'key.'),
                        {'$set': {'need_recalc': False}},
                    )

    @classmethod
    def get_representation(cls, sub_type, delay, deep):
        for representation in cls.instance.forecast_representations[sub_type]:
            if (representation.delay_from <= delay <= representation.delay_to and
                    representation.deep_from <= deep <= representation.deep_to):
                return representation


def save_forecast_parallel(args):
    """
    Функция для сохранения прогнозов, запускаемая параллельно в нескольких процессах.
    Инстанс Forecaster'а берем из Forecaster.instance, а из него берем прогнозы forecaster.thread_station_states.
    В функцию в качестве аргументов приходят индексы массива thread_station_states,
    которые должны быть обработаны данным процессом.
    """
    ind_from, ind_to = args
    forecaster = Forecaster.instance

    with log_run_time('updating tss parallel'):
        with BulkBuffer(ThreadStationState._get_collection(), max_buffer_size=2000, logger=log) as coll:
            items = list(forecaster.thread_station_states.items())[ind_from: ind_to]
            for key, events_states in items:
                # Выключаем прошлые прогнозы
                coll.update_many(
                    prefix_for_dict_keys(key.to_mongo(), 'key.'),
                    {'$set': {'outdated': True}}
                )

                # Добавляем новые прогнозы
                for tss in events_states:
                    tss_dict = tss.to_mongo()
                    tss_dict['outdated'] = False
                    coll.replace_one(
                        prefix_for_dict_keys(tss.key.to_mongo(), 'key.'),
                        tss_dict,
                        upsert=True,
                    )

                try:
                    log_suburban_events(forecaster.forecast_uid, now(), events_states)
                except Exception:
                    log.exception(u'Unable to save forecast log')

    return coll.operations_processed


def recalc_forecasts():
    Forecaster().run()


def get_no_data_events_states(th_event, rtstations):
    thread_station_states = []

    # Для первой станции событие "нет данных".
    ts_state = get_thread_station_state(th_event, rtstations[0])
    trigger_station = get_rtstation_key(rtstations[0])
    ts_state.set_departure_state(models.EventState(
        type=EventStateType.POSSIBLE_DELAY,
        thread_uid=rtstations[0].thread.uid,
        trigger=trigger_station,
        sub_type=EventStateSubType.NO_DATA_FIRST_STATION,
    ))
    thread_station_states.append(ts_state)

    # Считаем, что по всем остальным станциям поезд возможно опаздывает.
    for rts in rtstations[1:]:
        ts_state = get_possible_delay_no_data_state(th_event, rts, trigger_station)
        thread_station_states.append(ts_state)

    return thread_station_states


def compare_forecast_states(*states_lists):
    """
    Группируем possible_delay по rts и типу события.
    Для каждого rts берем любой из относящихся к нему tss.
    Заполняем события данного tss, выбирая "лучшее" среди сгруппированных.
    """
    states_by_rts = defaultdict(lambda: defaultdict(list))
    tss_list_by_rts = defaultdict(list)
    filtered_tss = []

    for tss in chain(*states_lists):
        for state_type in ['arrival_state', 'departure_state']:
            state = getattr(tss, state_type, None)
            if state and state.type == EventStateType.POSSIBLE_DELAY:
                states_by_rts[tss.rts][state_type].append(state)
                tss_list_by_rts[tss.rts].append(tss)

    for rts, rts_states in states_by_rts.items():
        tss = tss_list_by_rts[rts][0]
        for state_type, type_states in rts_states.items():
            setattr(tss, state_type, max(type_states, key=lambda x: x.forecast_delay))
        filtered_tss.append(tss)

    return filtered_tss


def compare_tsses_cancels_first(*states_lists):
    # type: (Iterable[ThreadStationState]) -> List[ThreadStationState]
    """
    Группируем tss по rts и типу события.
    Для каждого rts берём любой tss и заполняем отменой (если есть) или
    фактическим/прогнозируемым событием (если отмены нет)
    """
    states_by_rts = defaultdict(dict)
    tss_list_by_rts = defaultdict(list)
    filtered_tss = []

    for tss in chain(*states_lists):
        for state_type in ['arrival_state', 'departure_state']:
            state = getattr(tss, state_type, None)
            if state:
                if state.type == EventStateType.CANCELLED:
                    states_by_rts[tss.rts][state_type] = state
                    tss_list_by_rts[tss.rts].append(tss)
                else:
                    prev_state = states_by_rts[tss.rts].get(state_type)
                    if not prev_state or prev_state.type != EventStateType.CANCELLED:
                        states_by_rts[tss.rts][state_type] = state
                        tss_list_by_rts[tss.rts].append(tss)

    for rts, rts_states in states_by_rts.items():
        tss = tss_list_by_rts[rts][0]
        for state_type, state in rts_states.items():
            setattr(tss, state_type, state)
        filtered_tss.append(tss)

    return filtered_tss


def get_no_next_data_departure_state(th_event, delay):
    rts = th_event.last_event.rts
    departure_state, time_diff = None, None
    rts_arrival_msk_dt = rts.get_event_dt('arrival', th_event.start_dt, out_tz=MSK_TZ)
    if rts_arrival_msk_dt:
        rts_arrival_msk_dt = rts_arrival_msk_dt.replace(tzinfo=None)

    rts_departure_msk_dt = rts.get_event_dt('departure', th_event.start_dt, out_tz=MSK_TZ)
    if rts_departure_msk_dt:
        rts_departure_msk_dt = rts_departure_msk_dt.replace(tzinfo=None)

    if rts_departure_msk_dt and rts_arrival_msk_dt:
        minutes_diff = (rts_departure_msk_dt - rts_arrival_msk_dt).total_seconds() / 60
        time_diff = calc_stop_diff_time(minutes_diff)
        dep_delay, dep_dt, dep_deep = calc_rts_event_delay_params(th_event, delay, rts_departure_msk_dt, time_diff)

        if dep_delay >= conf.SUBURBAN_EVENT_PROPAGATION_DELTA:
            departure_state = get_possible_delay_event_state(
                th_event, rts, dep_delay, dep_dt, dep_deep, EventStateSubType.NO_NEXT_DATA
            )

    return departure_state, time_diff


def get_no_next_data_events_states(th_event, last_query_time, base_forecast_states, cancels):
    # строим факты для th_event
    thread_station_states, unknown_state_rts, _ = get_fact_states_unknown_rts(th_event, cancels)
    if not thread_station_states:
        return []

    delay = (last_query_time - th_event.trigger_station_event.dt_normative).total_seconds() / 60
    stop_minutes = 0

    if th_event.last_event_is_arrival and conf.SUBURBAN_NO_NEXT_DATA_ONLY_ARRIVAL:
        departure_state, time_diff = get_no_next_data_departure_state(th_event, delay)
        if time_diff:
            stop_minutes += time_diff
        if departure_state:
            thread_station_states[-1].set_departure_state(departure_state)

    # строим прогноз "застрял в пути"
    no_next_data_states = get_unknown_rtstations_possible_states(
        th_event, unknown_state_rts, stop_minutes, delay,
        get_no_next_data_possible_delay_state, conf.SUBURBAN_MIN_DELAY_FOR_NO_NEXT_DATA_EVENT
    )

    if not base_forecast_states:
        # если base_forecast не был построен, то пробуем посчитать прогнозы по фактам
        # в прогнозе по фактам возможен случай, когда в tss прибытие - факт, отправление - прогноз,
        # такой tss будет среди фактов (thread_station_states),
        # это не приводит к коллизиям т.к. прогноз "застрял в пути" строится со станции после последнего факта
        base_forecast_states = get_fact_unknown_states(th_event, unknown_state_rts, thread_station_states)
    if base_forecast_states:
        # сравниваем прогнозы по фактам и по "застрял в пути"
        no_next_data_states = compare_forecast_states(base_forecast_states, no_next_data_states)

    thread_station_states.extend(no_next_data_states)

    return thread_station_states


def get_unknown_rtstations_possible_states(th_event, unknown_state_rts, total_stop_minutes, delay,
                                           possible_delay_state_function, forecast_delay_const):
    """
    Функция генерирует прогнозируемые состояния для rts, которые идут после rts с фактическими данными.
    Для каждого rts вычисляется опоздание, исходя из времени, на которое могут быть сокращены остановки на всех
    предыдущих станциях, и времени, которое может нагнать поезд за время проделанного пути.
    :param th_event: События рассматриваемой нитки.
    :param unknown_state_rts: rts без данных.
    :param total_stop_minutes: Суммарное количество минут, на которое могут сократить остановки.
    :param delay: Опоздание в минутах.
    :param possible_delay_state_function: Функция, которая будет использована для создания состояний.
    :param forecast_delay_const: Константа для проверки существования рассчитываемого типа прогноза.
    :return: Список состояний для unknown_rtstations.
    """
    if not check_forecast_exists(delay, forecast_delay_const):
        return []

    thread_station_states = []
    for unknown_rts in unknown_state_rts:
        rts_arrival_msk_dt = unknown_rts.get_event_dt('arrival', th_event.start_dt, out_tz=MSK_TZ)
        if rts_arrival_msk_dt:
            rts_arrival_msk_dt = rts_arrival_msk_dt.replace(tzinfo=None)
            arr_delay, arr_dt, arr_deep = calc_rts_event_delay_params(
                th_event, delay, rts_arrival_msk_dt, total_stop_minutes
            )
        else:
            arr_delay, arr_dt, arr_deep = None, None, None

        rts_departure_msk_dt = unknown_rts.get_event_dt('departure', th_event.start_dt, out_tz=MSK_TZ)
        if rts_departure_msk_dt:
            rts_departure_msk_dt = rts_departure_msk_dt.replace(tzinfo=None)

        # total_stop_minutes нужно переcчитать до вычисления прогноза отправления с текущей станции,
        # чтобы учесть возможность сокращения остановки на этой станции.
        if rts_departure_msk_dt and rts_arrival_msk_dt:
            minutes_diff = (rts_departure_msk_dt - rts_arrival_msk_dt).total_seconds() / 60
            total_stop_minutes += calc_stop_diff_time(minutes_diff)

        if rts_departure_msk_dt:
            dep_delay, dep_dt, dep_deep = calc_rts_event_delay_params(
                th_event, delay, rts_departure_msk_dt, total_stop_minutes
            )
        else:
            dep_delay, dep_dt, dep_deep = None, None, None

        if rts_arrival_msk_dt == rts_departure_msk_dt:
            continue

        ts_state = possible_delay_state_function(
            th_event, unknown_rts, arr_dt, dep_dt, arr_delay, dep_delay, arr_deep, dep_deep
        )
        if ts_state:
            thread_station_states.append(ts_state)

    return thread_station_states


def calc_stop_diff_time(minutes_diff):
    return (minutes_diff - 1) if minutes_diff > 1 else 0


def calc_rts_event_delay_params(th_event, delay, event_time, total_stop_minutes):
    """
    Функция возвращает итоговое опоздание (с учетом нагона и времени остановок), прогнозируемое время события и
    глубину относительно станции последнего события.
    """
    delay -= total_stop_minutes
    deep = th_event.calc_last_event_deep(event_time) - total_stop_minutes
    hours_from_last_fact = deep / 60
    event_delay = delay - conf.SUBURBAN_REDUCTION_TIME_BY_HOUR * hours_from_last_fact
    event_dt = event_time + timedelta(minutes=event_delay)
    return event_delay, event_dt, deep


def get_possible_delay_no_data_state(th_event, rts, trigger_station):
    """
    Функция генерации состояний опозданий для событий типа no_data.
    :return: ts_state
    """
    ts_state = get_thread_station_state(th_event, rts)
    if rts.tz_arrival:
        ts_state.set_arrival_state(models.EventState(
            type=EventStateType.POSSIBLE_DELAY,
            thread_uid=rts.thread.uid,
            sub_type=EventStateSubType.NO_DATA,
            trigger=trigger_station
        ))

    if rts.tz_departure:
        ts_state.set_departure_state(models.EventState(
            type=EventStateType.POSSIBLE_DELAY,
            thread_uid=rts.thread.uid,
            sub_type=EventStateSubType.NO_DATA,
            trigger=trigger_station
        ))

    return ts_state


def get_no_next_data_possible_delay_state(th_event, rts, arr_dt, dep_dt, arr_delay, dep_delay, arr_deep, dep_deep):
    """
    Функция генерации состояний опозданий для событий типа no_next_data.
    :return: ts_state
    """
    if (arr_delay and arr_delay > conf.SUBURBAN_EVENT_PROPAGATION_DELTA) or \
            (dep_delay and dep_delay > conf.SUBURBAN_EVENT_PROPAGATION_DELTA):
        ts_state = get_thread_station_state(th_event, rts)

        if rts.tz_arrival and arr_delay >= conf.SUBURBAN_EVENT_PROPAGATION_DELTA:
            arrival_state = get_possible_delay_event_state(
                th_event, rts, arr_delay, arr_dt, arr_deep, EventStateSubType.NO_NEXT_DATA
            )
            ts_state.set_arrival_state(arrival_state)

        if rts.tz_departure and dep_delay >= conf.SUBURBAN_EVENT_PROPAGATION_DELTA:
            departure_state = get_possible_delay_event_state(
                th_event, rts, dep_delay, dep_dt, dep_deep, EventStateSubType.NO_NEXT_DATA
            )
            ts_state.set_departure_state(departure_state)

        return ts_state


def get_local_expected_event_time(event, rts):
    return MSK_TZ.localize(event.dt_normative).astimezone(rts.pytz).replace(tzinfo=None)


def get_threads_after_turnover_info(before_threads, today):
    numbers_after_date_station = []
    forecast_before_by_after = {}
    numbers_before = [turnover.number for (turnover, d, s, f, r) in before_threads]
    turnovers = TrainTurnover.objects.filter(number_before__in=numbers_before)
    turnover_by_number = defaultdict(list)
    for turnover in turnovers:
        turnover_by_number[turnover.number_before].append(turnover)
    for (thread, end_dt, station_key, forecast_dt, rts) in before_threads:
        end_date = end_dt.date()
        numb_turnovers = turnover_by_number.get(thread.number)
        if numb_turnovers:
            for numb_turnover in numb_turnovers:
                turnover_mask = RunMask(numb_turnover.year_days, today=today)
                dates = turnover_mask.dates()
                station_equals = str(numb_turnover.station.id) == station_key
                graph_equals = numb_turnover.graph == thread.schedule_plan
                if end_date in dates and station_equals and graph_equals:
                    numbers_after_date_station.append((numb_turnover.number_after, end_date, numb_turnover.station))
                    key_after = numb_turnover.number_after, end_date, station_key
                    thread.end_dt = end_dt
                    value_before = forecast_dt, rts, thread
                    forecast_before_by_after[key_after] = value_before
                    break

    return numbers_after_date_station, forecast_before_by_after


def find_thread_events_after_turnover(numbers_after_date_station, forecast_before_by_after, turnover_delta):
    # Оборот может "переходить через сутки".
    # При поиске ниток после оборота нужно учитывать нитки,
    # стартующие на следующий день после прибытия первой нитки.
    thread_events_keys = defaultdict(list)
    for (number_after, end_date, station) in numbers_after_date_station:
        suburban_key = get_thread_suburban_key(number_after, station)
        for (delta_days, d) in [(timedelta(days=0), end_date),
                                (timedelta(days=1), end_date + timedelta(days=1))]:
            thread_events_keys[delta_days].append({
                'thread_key': suburban_key,
                'date': str(d),
            })

    filtered_after_thread_events = []
    thread_before_by_th_event = {}
    for delta_days, events_keys in thread_events_keys.items():
        th_events = models.ThreadEvents.objects.aggregate(
            {'$project': {
                'tmp_key': {
                    'thread_key': '$key.thread_key',
                    'date': {'$dateToString': {'format': '%Y-%m-%d', 'date': '$key.thread_start_date'}}
                },
            }},
            {'$match': {'tmp_key': {'$in': events_keys}}}
        )
        th_events = models.ThreadEvents.objects.filter(id__in=[th_event['_id'] for th_event in th_events])

        for th_event in th_events:
            before_date = th_event.key.thread_start_date
            before_date -= delta_days
            number, station_key = get_suburban_key_number_station(th_event.suburban_key)
            key_after = (number, before_date.date(), station_key)
            before = forecast_before_by_after.get(key_after)
            if before:
                forecast_dt, rts, thread = before
                try:
                    first_expected_event = th_event.stations_expected_events[0]
                except IndexError:
                    log.error(u'Нет предполагаемых событий для нитки %s' % th_event)
                    continue
                event_time = get_local_expected_event_time(first_expected_event, rts)
                if (forecast_dt + turnover_delta) > event_time > thread.end_dt:
                    filtered_after_thread_events.append(th_event)
                    thread_before_by_th_event[th_event] = thread

    return filtered_after_thread_events, thread_before_by_th_event


def get_turnover_events_states(th_event, trigger):
    thread_station_states = []
    # Считаем, что при обороте поезд возможно опаздывает по всем станциям.
    for rts in th_event.rts_path:
        ts_state = get_thread_station_state(th_event, rts)
        if rts.tz_arrival is not None:
            ts_state.set_arrival_state(models.EventState(
                type=EventStateType.POSSIBLE_DELAY,
                thread_uid=rts.thread.uid,
                sub_type=EventStateSubType.TRAIN_TURNOVER,
                trigger=trigger
            ))

        if rts.tz_departure is not None:
            ts_state.set_departure_state(models.EventState(
                type=EventStateType.POSSIBLE_DELAY,
                thread_uid=rts.thread.uid,
                sub_type=EventStateSubType.TRAIN_TURNOVER,
                trigger=trigger,
            ))

        thread_station_states.append(ts_state)

    return thread_station_states


def get_companies_crashes():
    crash_dt_by_company = {}

    for crash in models.CompanyCrash.objects.all():
        crash_dt_by_company[crash.company] = crash

    return crash_dt_by_company


def get_forecast_representations():
    representation_by_type = defaultdict(list)

    for representation in ForecastRepresentation.objects.all():
        representation_by_type[representation.type].append(representation)

    return representation_by_type


def prepare_cancels(th_event):
    # type: (ThreadEvents) -> (List[ThreadStationState], bool)

    stations_cancels = sorted(th_event.stations_cancels, key=lambda x: x.dt_save, reverse=True)[0]
    if not stations_cancels.cancelled_stations:
        log.info('annulled cancel: {}'.format(th_event.suburban_key))
        return [], False

    rtss_by_id = {
        (get_rtstation_key(rts), rts.tz_departure, rts.tz_arrival): rts
        for rts in th_event.rts_path
    }
    cancelled_rtstations = [
        rtss_by_id.get(cancelled_station.matching_key, None)
        for cancelled_station in stations_cancels.cancelled_stations
    ]

    if None in cancelled_rtstations:
        log.warning('there are unmatched cancelled stations in thread {}: {} -> {}'.format(
            th_event.suburban_key,
            [cancelled_station.matching_key for cancelled_station in stations_cancels.cancelled_stations],
            list(rtss_by_id.keys())
        ))
        return [], False
    first_cancelled_rtstation, last_cancelled_rtstation = cancelled_rtstations[0], cancelled_rtstations[-1]

    thread_station_states = []
    for i, rts in enumerate(cancelled_rtstations):
        tss = get_thread_station_state(th_event, rts)
        if rts != first_cancelled_rtstation:
            tss.set_arrival_state(models.EventState(
                type=EventStateType.CANCELLED,
                thread_uid=rts.thread.uid,
                sub_type=EventStateSubType.MOVISTA_CANCEL
            ))
        if rts != last_cancelled_rtstation:
            tss.set_departure_state(models.EventState(
                type=EventStateType.CANCELLED,
                thread_uid=rts.thread.uid,
                sub_type=EventStateSubType.MOVISTA_CANCEL
            ))
        thread_station_states.append(tss)

    return thread_station_states, len(cancelled_rtstations) == len(th_event.rts_path)


def get_rtstations_for_no_data(th_event, cancels, now_dt):
    # type: (ThreadEvents, List[ThreadStationState], datetime) -> List[RTStation]

    rtstations = th_event.rts_path

    if not cancels:
        return rtstations

    if cancels[0].rts.tz_arrival is not None:
        return rtstations

    if th_event.start_dt + timedelta(minutes=cancels[-1].rts.tz_departure) < now_dt:
        return rtstations[len(cancels)-1:]

    return []
