# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import os
import random
from datetime import datetime

from django.conf import settings
from django.utils.encoding import force_bytes
from mongoengine.base import BaseDocument

from travel.library.python.logbroker.writer import LogbrokerWriter

from common.settings.configuration import Configuration
from common.settings.utils import define_setting
from common.utils.date import MSK_TZ
from common.utils.marketstat import JsonLog


log = logging.getLogger(__name__)

suburban_events_forecast_log = JsonLog(settings.SUBURBAN_EVENTS_FORECAST_LOG)
rasp_suburban_raw_rzd_events_log = JsonLog(settings.RAW_RZD_EVENTS_LOG)

DT_FORMAT = '%Y-%m-%d %H:%M:%S'


define_setting('SUBURBAN_EVENTS_FORECAST_LOGBROKER_TOPIC', {
    Configuration.PRODUCTION: '/rasp-front/rasp-suburban-events-forecast-log',
    Configuration.TESTING: '/rasp-front/rasp-suburban-testing-events-forecast-log',
}, default='/rasp-front/rasp-suburban-testing-events-forecast-log')


define_setting('SUBURBAN_EVENTS_WRITE_FORECAST_LOGS_TO_YT', default=False)


def get_event_time_str(rts, event_state_name, thread_start_date):
    event_dt = rts.get_event_dt(
        event_state_name,
        datetime.combine(thread_start_date, rts.thread.tz_start_time),
        out_tz=MSK_TZ
    )
    return event_dt.strftime(DT_FORMAT) if event_dt else None


def create_log_broker_writer():
    if not settings.SUBURBAN_EVENTS_WRITE_FORECAST_LOGS_TO_YT:
        log.debug('Skip write to Logbroker. Disabled by config')
        return

    token = settings.LOGBROKER_OAUTH_TOKEN
    if not token:
        log.debug('Skip write to Logbroker. No oauth token provided.')
        return

    topic = settings.SUBURBAN_EVENTS_FORECAST_LOGBROKER_TOPIC
    source_id = 'log_suburban_events_{}_{}_{}'.format(os.getcwd().__hash__(), os.getpid(), random.randint(0, 100000))

    return LogbrokerWriter(token=str(token), topic=str(topic), source_id=str(source_id))


def log_suburban_events(forecast_uid, save_time, events):
    """
    Логируем данные прогнозатора для отправки в YT
    https://st.yandex-team.ru/RASPEXPORT-251
    """

    writer = create_log_broker_writer()
    if not writer:
        return

    with writer:
        for event in events:
            rts = event.rts
            try:
                thread_path = event.th_event.rts_path
                start_date = event.key.thread_start_date

                for event_state_name in ['arrival', 'departure']:
                    event_state = getattr(event, event_state_name + '_state', None)
                    if event_state is None:
                        continue

                    data = {
                        'eventtime': save_time.strftime(DT_FORMAT),
                        'forecast_uid': forecast_uid,
                        'thread_key': event.key.thread_key,
                        'thread_start_date': start_date.strftime('%Y-%m-%d'),
                        'first_station_departure': get_event_time_str(thread_path[0], 'departure', start_date),
                        'last_station_arrival': get_event_time_str(thread_path[-1], 'arrival', start_date),
                        'thread_title': rts.thread.title,
                        'thread_uid': rts.thread.uid,
                        'station_key': event.key.station_key,
                        'station_title': rts.station.title,
                        'action_type': event_state_name,
                        'event_type': event_state.type,
                        'event_normative_time': get_event_time_str(rts, event_state_name, start_date),
                        'departure_subdir': rts.departure_subdir,
                        'event_cause': {}
                    }

                    if event_state.dt:
                        data['event_fact_time'] = (rts.station.pytz.localize(event_state.dt).
                                                   astimezone(MSK_TZ).replace(tzinfo=None).strftime(DT_FORMAT))

                    if event_state.forecast_dt:
                        data['event_forecast_time'] = (rts.station.pytz.localize(event_state.forecast_dt).
                                                       astimezone(MSK_TZ).replace(tzinfo=None).strftime(DT_FORMAT))

                    if event_state.minutes_from is not None:
                        data['event_minutes_from'] = event_state.minutes_from

                    if event_state.minutes_to is not None:
                        data['event_minutes_to'] = event_state.minutes_to

                    if event_state.sub_type or event_state.trigger:
                        data['event_cause'].update({
                            'sub_type': event_state.sub_type,
                            'trigger': event_state.trigger,
                        })

                    if event_state.time_from_delay_station is not None:
                        data['event_cause']['time_from_delay_station'] = event_state.time_from_delay_station

                    if event_state.original_event:
                        data['event_rzd_normative_time'] = event_state.original_event.dt_normative.strftime(DT_FORMAT)

                    if rts.tz_arrival == rts.tz_departure:
                        data['no_stop'] = True

                    writer.write(force_bytes(json.dumps(data, ensure_ascii=False)))
            except Exception:
                log.exception(u'Unable to log event {}, rts {}, thread {}'.format(event, rts.id, event.rts.thread))


def log_rzd_raw_data(fetch_uid, save_time, rows_dicts):
    eventtime = save_time.strftime(DT_FORMAT)

    for row in rows_dicts:
        prepared_row = {
            'eventtime': eventtime,
            'fetch_uid': fetch_uid,
        }
        for k, v in row.items():
            if isinstance(v, datetime):
                v = v.strftime(DT_FORMAT)
            elif isinstance(v, BaseDocument):
                continue

            prepared_row[k] = v

        rasp_suburban_raw_rzd_events_log.log(prepared_row)
