# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from functools import reduce
from operator import xor

import six
from django.conf import settings
from django.db import models
from mongoengine import EmbeddedDocument
from mongoengine.fields import (
    IntField, FloatField, StringField, DateTimeField, EmbeddedDocumentField, ReferenceField,
    EmbeddedDocumentListField, BooleanField, ListField, DateField
)

from common.apps.suburban_events.utils import EventStateType, EventStateSubType, get_rtstation_key
from common.db.mongo.base import RaspDocument


class SuburbanKey(models.Model):
    thread = models.OneToOneField('www.RThread', null=False, related_name='suburban_key', db_index=True)
    key = models.CharField(max_length=30, null=False, db_index=True)


class ForecastRepresentation(models.Model):
    TYPE_CHOICES = (
        (EventStateSubType.NO_NEXT_DATA, 'поезд застрял в пути'),
        (EventStateSubType.FACT, 'по фактическим данным'),
    )

    type = models.CharField(
        verbose_name='Тип прогноза', default=EventStateSubType.FACT, choices=TYPE_CHOICES, max_length=50
    )
    delay_from = models.IntegerField(verbose_name='Время опоздания (начало интервала)')
    delay_to = models.IntegerField(verbose_name='Время опоздания (конец интервала)')
    deep_from = models.IntegerField(verbose_name='Глубина прогноза (начало интервала)')
    deep_to = models.IntegerField(verbose_name='Глубина прогноза (конец интервала)')
    minutes_from = models.IntegerField(
        verbose_name='Представление прогноза (начало интервала)', default=None, null=True, blank=True
    )
    minutes_to = models.IntegerField(
        verbose_name='Представление прогноза (конец интервала)', default=None, null=True, blank=True
    )

    class Meta:
        verbose_name = 'Представление прогноза'
        verbose_name_plural = 'Представления прогнозов'


@six.python_2_unicode_compatible
class LVGD01_TR2PROC_query(RaspDocument):
    meta = {
        'collection': 'lvgd01_tr2proc_query',
        'verbose_name': 'Электрички realtime - лог запросов',
        'indexes': ['queried_at'],
        'strict': False,
        'index_background': True,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    queried_at = DateTimeField(required=True, help_text='''Во сколько сделан запрос (Europe/Moscow)''')
    time_taken = IntField(default=None, help_text='''Как долго выполнялся запрос''')
    query_from = DateTimeField(required=True, help_text='''Параметр DTFROM, переданный при запросе''')
    query_to = DateTimeField(required=True, help_text='''Параметр DTTO, переданный при запросе''')
    rows_count = IntField(default=None, help_text='''Количество строк в ответе''')
    new_rows_count = IntField(default=None, help_text='''
        Количество строк в ответе, не являющихся дубликатами предыдущего запроса
    ''')
    tries_count = IntField(default=None, help_test='''Количество попыток получить данные от РЖД''')
    exception = StringField(help_text='''Текст ошибки, если была''')
    source = StringField(help_text='''Заполняется, если данный запрос был загружен, а не создан''')

    def __str__(self):
        return '{} - {}'.format(self.query_from.strftime('%Y-%m-%d %H:%M'), self.query_to.strftime('%Y-%m-%d %H:%M'))

    def to_dict(self):
        return {f: getattr(self, f) for f in self._fields_ordered}


@six.python_2_unicode_compatible
class LVGD01_TR2PROC_base(RaspDocument):
    meta = {
        'abstract': True,
        'strict': False,
        'index_background': True,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    class Sources(object):
        # SOURCE field values
        ASOUP = 1
        GID_URAL = 2
        SAI_PS = 3
        GIS_RZD = 4

    ID_TRAIN = IntField(required=True, help_text='''
        Идентификатор отправившегося поезда - все его события будут иметь один ID_TRAIN.
        Когда этот же поезд отправится завтра - ID_TRAIN поменяется
        Оригинальное название колонки в базе РЖД - ID
    ''')
    IDTR = IntField(required=True, help_text='''
        Идентификатор поезда из НГДП
    ''')
    IDRASP = IntField(required=True, help_text='''
        Идентификатор расписания назначенного поезда из НГДП
    ''')
    STORASP = IntField(required=True, help_text='''
        Код станции отправления
    ''')
    STOEX = IntField(required=True, help_text='''
        Код станции отправления ЭКСПРЕСС
    ''')
    NAMESTO = StringField(required=True, max_length=100, help_text='''
        Наименование станции отправления
    ''')
    STNRASP = IntField(required=True, help_text='''
        Код станции назначения
    ''')
    STNEX = IntField(required=True, help_text='''
        Код станции назначения ЭКСПРЕСС
    ''')
    NAMESTN = StringField(required=True, max_length=100, help_text='''
        Наименование станции назначения
    ''')
    NOMPEX = StringField(required=True, max_length=100, help_text='''
        № поезда в системе ЭКСПРЕСС
    ''')
    NAMEP = StringField(required=True, max_length=100, help_text='''
        Наименование поезда
    ''')
    SOURCE = IntField(required=True, help_text='''
        Система – источник информации. 1 – АСОУП, 2 – ГИД Урал, 3 – САИ ПС, 4 – ГИС РЖД
    ''')
    KODOP = IntField(required=True, help_text='''
        Код операции с поездом. 1 – прибытие, 3 – отправление
    ''')
    DOR = IntField(required=True, help_text='''
        Код дороги ввода информации
    ''')
    OTD = IntField(required=True, help_text='''
        Код отделения
    ''')
    NOMRP = IntField(required=True, help_text='''
        Порядковый номер станции по ходу поезда
    ''')
    STOPER = IntField(required=True, help_text='''
        Код станции совершения операции
    ''')
    STOPEREX = IntField(required=True, help_text='''
        Код станции совершения операции ЭКСПРЕСС
    ''')
    STNAME = StringField(required=True, max_length=100, help_text='''
        Наименование станции совершения операции
    ''')
    TIMEOPER_N = DateTimeField(required=True, help_text='''
        Дата и время совершения операции с поездом из эталонного расписания
    ''')
    TIMEOPER_F = DateTimeField(required=True, help_text='''
        Дата и время совершения операции с поездом фактические
    ''')
    KM = IntField(required=True, help_text='''
        Километраж от начала маршрута
    ''')
    PRSTOP = FloatField(required=True, help_text='''
        Признак стоянки без посадки/высадки пассажиров.
        1-стоянка без посадки/высадки; 0–обычный режим
        Признак стоянки без посадки/высадки пассажиров означает следующее: если время стоянки не равно 0 и признак 0 –
        посадка/высадка есть, признак 1 – посадки/высадки нет; если время стоянки равно 0, признак не надо анализировать
    ''')
    PRIORITY = FloatField(required=True, help_text='''
        Приоритетная запись о фактическом маршруте
    ''')
    PRIORITY_RATING = FloatField(required=True, help_text='''
        Рейтинг системы-источника в таблице приоритетов
    ''')

    @classmethod
    def fields_list(cls):
        return LVGD01_TR2PROC_base._fields_ordered

    def __str__(self):
        return '{} {} {} {} {} {} {}'.format(
            self.ID_TRAIN, self.NAMESTO, self.NAMESTN, self.STNAME, self.KODOP, self.SOURCE, self.TIMEOPER_F)

    def __eq__(self, other):
        return all(
            getattr(self, field) == getattr(other, field) for field in self.fields_list()
        )

    def __lt__(self, other):
        for field in self.fields_list():
            self_value, other_value = getattr(self, field), getattr(other, field)
            if self_value < other_value:
                return True
            elif self_value > other_value:
                return False

        return False

    def to_dict(self):
        return {f: getattr(self, f) for f in self._fields_ordered}


class LVGD01_TR2PROC(LVGD01_TR2PROC_base):
    """ Сырые данные о событиях от РЖД (процедура LVGD01_TR2PROC), разбитые по запросам.
        Коллекция может содержать дубли событий, т.к. запросы делаются с перекрытием временных интервалов.
    """

    meta = {
        'collection': 'lvgd01_tr2proc',
        'verbose_name': 'Электрички realtime - позапросно',
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    query = ReferenceField(LVGD01_TR2PROC_query, required=True)


class LVGD01_TR2PROC_feed(LVGD01_TR2PROC_base):
    """ Лента событий от РЖД на основе сырых данных из LVGD01_TR2PROC, очищенная от дублей"""

    meta = {
        'collection': 'lvgd01_tr2proc_feed',
        'verbose_name': 'Электрички realtime - лента (mongo)',
        'strict': False,
        'indexes': ['TIMEOPER_F', 'query'],
        'index_background': True,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    query = ReferenceField(LVGD01_TR2PROC_query)


@six.python_2_unicode_compatible
class EventState(EmbeddedDocument):
    meta = {'strict': False}

    type = StringField(choices=(
        (EventStateType.FACT, 'Фактическое время'),
        (EventStateType.FACT_INTERPOLATED, 'Интерполированное время'),
        (EventStateType.POSSIBLE_DELAY, 'Возможно опоздание'),
        (EventStateType.POSSIBLE_OK, 'Ожидается по расписанию'),
        (EventStateType.CANCELLED, 'Отменён'),
        (EventStateType.UNDEFINED, 'Состояние неизвестно')
    ))

    # точное время состояния
    dt = DateTimeField()

    # прогнозное время
    forecast_dt = DateTimeField()
    forecast_delay = FloatField()

    # интервал состояния (например, "опаздывает на 15-20 минут")
    minutes_from = IntField()
    minutes_to = IntField()

    # uid треда, для которго построен прогноз
    thread_uid = StringField()

    # Для правильного отображения причины опоздания.
    sub_type = StringField()
    trigger = StringField()
    time_from_delay_station = IntField()

    def __str__(self):
        if self.type == EventStateType.FACT:
            tail = ' '
            if self.minutes_from > 0:
                tail += 'late {}'.format(self.minutes_from)
            elif self.minutes_from < 0:
                tail += 'early {}'.format(abs(self.minutes_from))

            return '{} {}{}'.format(self.type, self.dt, tail)
        elif self.type == EventStateType.POSSIBLE_DELAY:
            return '{} {} - {}'.format(self.type, self.minutes_from, self.minutes_to)
        else:
            return '{}'.format(self.type)

    @property
    def original_event(self):
        return getattr(self, '_original_event', None)

    @original_event.setter
    def original_event(self, original_event_):
        self._original_event = original_event_

    @property
    def tss(self):
        return getattr(self, '_tss')

    @tss.setter
    def tss(self, tss_):
        self._tss = tss_


@six.python_2_unicode_compatible
class ThreadKey(EmbeddedDocument):
    meta = {'strict': False}

    thread_key = StringField(required=True, help_text='Номер нитки')
    thread_start_date = DateTimeField(required=True, help_text='Дата старта нитки')
    thread_type = StringField()
    clock_direction = IntField()

    def __str__(self):
        return 'TKey({} {})'.format(self.thread_key, self.thread_start_date)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        hash_str = '{}_{}_{}_{}'.format(self.thread_key, self.thread_start_date, self.thread_type, self.clock_direction)
        return hash(hash_str)


@six.python_2_unicode_compatible
class ThreadState(RaspDocument):
    key = EmbeddedDocumentField(ThreadKey)
    state = EmbeddedDocumentField(EventState, required=True)
    tz = StringField()  # временная зона нитки

    meta = {
        'indexes': [
            {
                'fields': ['key'],
                'unique': True,
            },
        ],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return 'TSS({})'.format(self.key)

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class ThreadStationKey(EmbeddedDocument):
    thread_key = StringField(required=True, help_text='Номер нитки')
    thread_start_date = DateTimeField(required=True, help_text='Дата старта нитки')
    station_key = StringField(required=True, help_text='Идентификатор станции в нитке.')
    thread_type = StringField()
    clock_direction = IntField()
    arrival = IntField()
    departure = IntField()

    def __str__(self):
        return 'TSKey({} {} {})'.format(self.thread_key, self.thread_start_date, self.station_key)

    def __repr__(self):
        return self.__str__()

    def __hash__(self):
        hash_str = '{}_{}_{}_{}_{}'.format(self.thread_key, self.thread_start_date, self.station_key,
                                           self.thread_type, self.clock_direction)
        return hash(hash_str)

    meta = {'strict': False}


@six.python_2_unicode_compatible
class ThreadStationState(RaspDocument):
    key = EmbeddedDocumentField(ThreadStationKey)
    arrival = IntField()
    departure = IntField()
    tz = StringField()  # временная зона станции
    arrival_state = EmbeddedDocumentField(EventState, required=False)
    departure_state = EmbeddedDocumentField(EventState, required=False)
    passed_several_times = BooleanField()
    outdated = BooleanField(required=False)

    meta = {
        'indexes': [
            ['key'],
            {
                'fields': ['key.thread_key', 'key.thread_start_date', 'key.thread_type', 'key.clock_direction'],
                'name': 'thread_index'
            }
        ],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return 'TSS({}) {} {}'.format(self.key, self.arrival, self.departure)

    def __repr__(self):
        return self.__str__()

    @property
    def rts(self):
        return self._rts

    @rts.setter
    def rts(self, rts_):
        self._rts = rts_

    @property
    def th_event(self):
        return self._th_event

    @th_event.setter
    def th_event(self, th_event_):
        self._th_event = th_event_

    def set_departure_state(self, event_state):
        self.departure_state = event_state
        event_state.tss = self

    def set_arrival_state(self, event_state):
        self.arrival_state = event_state
        event_state.tss = self


@six.python_2_unicode_compatible
class StationEvent(EmbeddedDocument):
    station_key = StringField(required=True, null=False)
    type = StringField(required=True, null=False)  # arrival/departure

    dt_normative = DateTimeField(required=True, null=False)
    dt_fact = DateTimeField(required=True, null=False)
    time = IntField(required=True)  # в минутах от старта нитки

    dt_save = DateTimeField()  # время сохранения событий из одного пакета ржд

    # идентификатор конкретной станции для случаев, когда наша нитка проходит одну станцию 2 раза
    twin_key = StringField(required=True)
    weight = FloatField(required=True, null=False)  # вес события (для событий из разных источников)
    passed_several_times = BooleanField()  # проходит ли нитка такую станцию более чем 1 раз

    def __str__(self):
        return '{} {: <9} {} ({})  {}'.format(
            self.station_key,
            self.type,
            self.dt_fact,
            self.dt_normative,
            self.weight
        )

    meta = {'strict': False}

    # разница между фактическим событием и нормативным временем
    # minutes_diff > 0 - опоздание, minutes_diff < 0 - опережение
    @property
    def minutes_diff(self):
        return getattr(self, '_minutes_diff')

    @minutes_diff.setter
    def minutes_diff(self, minutes_diff_):
        self._minutes_diff = minutes_diff_

    @property
    def station_idx(self):
        return getattr(self, '_station_idx')

    @station_idx.setter
    def station_idx(self, station_idx_):
        self._station_idx = station_idx_

    @property
    def rts(self):
        return getattr(self, '_rts')

    @rts.setter
    def rts(self, rts_):
        self._rts = rts_

    # нормативное время события из нашей базы
    @property
    def rts_dt_normative(self):
        return getattr(self, '_rts_dt_normative')

    @rts_dt_normative.setter
    def rts_dt_normative(self, rts_dt_normative_):
        self._rts_dt_normative = rts_dt_normative_


@six.python_2_unicode_compatible
class StationExpectedEvent(EmbeddedDocument):
    station_key = StringField(required=True, null=False)
    type = StringField(required=True, null=False)

    dt_normative = DateTimeField(required=True, null=False)
    time = IntField(required=True)
    passed_several_times = BooleanField(required=True, default=False)

    def __str__(self):
        return '{} {: <9} {}'.format(
            self.station_key,
            self.type,
            self.dt_normative,
        )

    meta = {
        'indexes': [
            {
                'fields': ['dt_normative', 'station_key', 'type'],
            },
        ],
        'index_background': True,
        'strict': False,
    }


class CancelledStation(EmbeddedDocument):
    station_key = StringField(required=True, null=False)
    departure_time = IntField()
    arrival_time = IntField()

    @classmethod
    def create_from_rts(cls, rts):
        return cls(
            station_key=get_rtstation_key(rts),
            departure_time=rts.tz_departure,
            arrival_time=rts.tz_arrival
        )

    @property
    def matching_key(self):
        return self.station_key, self.departure_time, self.arrival_time


class StationCancel(EmbeddedDocument):
    cancelled_stations = EmbeddedDocumentListField(CancelledStation)
    rtstations = ListField(IntField(required=True, null=False))  # TODO: оставлено для обратной совместимости
    dt_save = DateTimeField(required=True, null=False)


class ThreadEvent(EmbeddedDocument):
    pass


@six.python_2_unicode_compatible
class ThreadEvents(RaspDocument):
    key = EmbeddedDocumentField(ThreadKey, required=True, null=False)
    thread_events = EmbeddedDocumentListField(ThreadEvent)
    stations_events = EmbeddedDocumentListField(StationEvent)
    stations_cancels = EmbeddedDocumentListField(StationCancel)
    stations_expected_events = EmbeddedDocumentListField(StationExpectedEvent)
    last_station_event = EmbeddedDocumentField(StationEvent)
    need_recalc = BooleanField(required=True, null=False, default=True)
    trigger_station_event = EmbeddedDocumentField(StationEvent)
    thread_company = IntField()
    meta = {
        'indexes': [
            {
                'fields': ['key.thread_key', 'key.thread_start_date', 'key.thread_type', 'key.clock_direction'],
                'unique': True,
                'name': 'key'
            },
            {
                'fields': ['stations_expected_events.dt_normative',
                           'stations_expected_events.station_key',
                           'stations_expected_events.type'],
                'name': 'expected_events_index'
            },
            'need_recalc',
            'stations_cancels'
        ],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return 'TE({}) {} {}'.format(self.key, len(self.thread_events), len(self.stations_events))

    def __repr__(self):
        return self.__str__()

    def calc_last_event_deep(self, msk_tz_time):
        return (msk_tz_time - self.last_event.rts_dt_normative).total_seconds() / 60

    @property
    def start_dt(self):
        return self.key.thread_start_date

    @property
    def suburban_key(self):
        return self.key.thread_key

    @property
    def thread(self):
        return getattr(self, '_thread', None)

    @thread.setter
    def thread(self, _thread):
        self._thread = _thread

    @property
    def rts_path(self):
        return getattr(self, '_rts_path', None)

    @rts_path.setter
    def rts_path(self, _rts_path):
        self._rts_path = _rts_path

    @property
    def last_event(self):
        return getattr(self, '_last_event')

    @last_event.setter
    def last_event(self, _last_event):
        self._last_event = _last_event

    @property
    def last_event_is_arrival(self):
        return getattr(self, '_is_arrival')

    @last_event_is_arrival.setter
    def last_event_is_arrival(self, _is_arrival):
        self._is_arrival = _is_arrival


@six.python_2_unicode_compatible
class StationUpdateInfo(RaspDocument):
    station_key = StringField(required=True, null=False)
    timestamp = DateTimeField()

    meta = {
        'indexes': [
            'timestamp',
            {
                'fields': ['station_key'],
                'unique': True,
            },
        ],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return 'SK({})'.format(self.station_key)

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class UpdatedThread(RaspDocument):
    uid = StringField(required=True, null=False, unique_with='start_date')
    start_date = DateTimeField(required=True, null=False)

    meta = {
        'indexes': ['uid'],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return '{}__{}'.format(self.uid, self.start_date)

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class CompanyCrash(RaspDocument):
    company = IntField(required=True, null=False, unique=True)
    first_dt = DateTimeField(required=True, null=False)
    last_dt = DateTimeField()
    first_rate = FloatField()
    last_rate = FloatField()
    avr_rate = FloatField()

    meta = {
        'indexes': ['company'],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return '{}__{}'.format(self.company, self.dt.isoformat())

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class HourEventsRate(RaspDocument):
    company = IntField(required=True, null=False)
    hour = IntField(required=True, null=False)
    rate = FloatField()

    meta = {
        'indexes': [
            {
                'fields': ['company', 'hour'],
                'name': 'company_hour_index'
            },
        ],
        'index_background': True,
        'strict': False,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    def __str__(self):
        return '{}__{}'.format(self.company, self.hour)

    def __repr__(self):
        return self.__str__()


@six.python_2_unicode_compatible
class MovistaCancelRaw(RaspDocument):
    meta = {
        'collection': 'movista_cancel_raw',
        'verbose_name': 'Отмена ЦППК из Мовисты (сырые данные)',
        'strict': False,
        'indexes': ['create_dt', 'departure_date'],
        'index_background': True,
        'db_alias': settings.SUBURBAN_EVENTS_DATABASE_NAME,
    }

    create_dt = DateTimeField(required=True, null=False)
    departure_date = DateField(required=True, null=False)
    train_number = StringField(required=True, null=False)
    start_express_id = IntField(required=True, null=False)
    finish_express_id = IntField(required=True, null=False)
    from_express_id = IntField(null=True)
    to_express_id = IntField(null=True)

    def __str__(self):
        return '"{}" {} ({}), {}->{} ({}->{})'.format(
            self.train_number, self.departure_date, self.create_dt,
            self.start_express_id, self.finish_express_id,
            self.from_express_id, self.to_express_id
        )

    def __repr__(self):
        return self.__str__()

    @property
    def _compare_fields(self):
        return [
            self.create_dt, self.departure_date, self.train_number,
            self.start_express_id, self.finish_express_id,
            self.from_express_id, self.to_express_id
        ]

    def __hash__(self):
        return reduce(xor, map(hash, self._compare_fields))

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return all(x == y for (x, y) in zip(self._compare_fields, other._compare_fields))
