# coding: utf8
import logging

from django.db import transaction

from common.apps.suburban_events.models import SuburbanKey
from common.apps.suburban_events.utils import get_thread_suburban_key
from common.models.schedule import RTStation
from common.models.transport import TransportType
from travel.rasp.library.python.common23.logging import log_run_time

log = logging.getLogger(__name__)


@transaction.atomic
def calc_suburban_keys():
    with log_run_time('get rtstations', logger=log):
        rtstations = list(RTStation.objects.filter(
            thread__t_type=TransportType.SUBURBAN_ID,
            tz_arrival=None,  # только станции отправления
        ).select_related(
            'thread', 'station'
        ).only(
            'thread_id', 'station_id', 'thread__number'
        ))

    with log_run_time('save {} suburban keys'.format(len(rtstations)), logger=log):
        for rts in rtstations:
            create_suburban_key_from_first_rts(rts)


def create_suburban_key_from_first_rts(first_rts):
    """
    Создание и сохранение SuburbanKey для нитки
    :param first_rts: Первая станция нитки
    """
    SuburbanKey.objects.update_or_create(
        thread=first_rts.thread,
        defaults=dict(
            key=get_thread_suburban_key(first_rts.thread.number, first_rts.station)
        )
    )
