# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict
from functools import partial

from common.apps.suburban_events import dynamic_params
from common.apps.suburban_events.forecast.match_cppk import CppkMatcher
from common.apps.suburban_events.models import CancelledStation, MovistaCancelRaw, StationCancel, ThreadEvents
from common.db.mongo.bulk_buffer import BulkBuffer
from travel.rasp.library.python.common23.logging import log_run_time

log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def get_new_movista_raw_cancels(last_cancel_id):
    query = MovistaCancelRaw.objects.all().order_by('id')
    if last_cancel_id:
        query = query.filter(id__gt=last_cancel_id)
    return list(query.aggregate())


def save_matched_cancels(cancels):
    # type: (List[CppkEvent]) -> None
    with log_run_time('update thread events with movista cancels'):
        cancel_by_tkey = defaultdict(list)
        for cancel in cancels:
            cancel_by_tkey[cancel.thread_key].append(cancel)

        with BulkBuffer(ThreadEvents._get_collection(), max_buffer_size=200, logger=log) as coll:
            for tkey, tkey_cancels in cancel_by_tkey.items():
                stations_cancels = [
                    StationCancel(
                        cancelled_stations=list(map(CancelledStation.create_from_rts, cancel.cancelled_stations)),
                        dt_save=cancel.create_dt
                    ).to_mongo() for cancel in sorted(tkey_cancels, key=lambda x:x.create_dt)
                ]
                coll.update_one(
                    {
                        'key.thread_key': tkey.thread_key,
                        'key.thread_start_date': tkey.thread_start_date,
                        'key.thread_type': tkey.thread_type,
                        'key.clock_direction': tkey.clock_direction
                    },
                    {
                        '$set': {'need_recalc': True},
                        '$push': {'stations_cancels': {'$each': stations_cancels}},
                    },
                    upsert=True,
                )


def update_movista_matched_cancels():
    with log_run_time('update movista matched cancels'):
        last_cancel_id = dynamic_params.get_param('last_movista_raw_cancel_id')
        raw_cancels = get_new_movista_raw_cancels(last_cancel_id)
        if not raw_cancels:
            log.info('no new movista raw cancels')
            return
        log.info('{} new movista raw cancels'.format(len(raw_cancels)))

        matcher = CppkMatcher(raw_cancels, log)
        matched_cancels = matcher.match()
        log.info('{} matched movista cancels'.format(len(matched_cancels)))

        save_matched_cancels(matched_cancels)
        dynamic_params.set_param('last_movista_raw_cancel_id', raw_cancels[-1]['_id'])
