# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, timedelta

import logging
from functools import partial
from typing import Dict, List, Optional

import requests

from common.apps.suburban_events.models import MovistaCancelRaw
from common.data_api.movista.instance import movista_client
from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date.environment import now
from travel.rasp.library.python.common23.logging import log_run_time


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def cancel_to_db_raw(cancel):
    # type: (Dict[unicode, unicode]) -> MovistaCancelRaw

    # RASPINFRA-4931
    try:
        create_dt = datetime.strptime(cancel['createDate'], '%Y-%m-%dT%H:%M:%S.%f')
    except ValueError:
        create_dt = datetime.strptime(cancel['createDate'], '%Y-%m-%dT%H:%M:%S')

    return MovistaCancelRaw(
        create_dt=create_dt,
        departure_date=datetime.strptime(cancel['date'], '%Y-%m-%d').date(),
        train_number=cancel['trainNumber'],
        start_express_id=cancel['startExpressId'],
        finish_express_id=cancel['finishExpressId'],
        from_express_id=cancel['fromExpressId'],
        to_express_id=cancel['toExpressId']
    )


def create_annulled_cancel(prev_cancel):
    # type: (MovistaCancelRaw) -> MovistaCancelRaw
    """
    Если при новом запросе к Мовисте перестала приходить отмена по нитке - создаём "отмену" отмены.
    Нужно как триггер для пересчёта нитки прогнозатором без предыдущих отмен.
    """

    return MovistaCancelRaw(
        create_dt=now(),
        departure_date=prev_cancel.departure_date,
        train_number=prev_cancel.train_number,
        start_express_id=prev_cancel.start_express_id,
        finish_express_id=prev_cancel.finish_express_id,
        from_express_id=None,
        to_express_id=None
    )


def fetch_new_movista_cancels():
    # type: () -> Optional[List[MovistaCancelRaw]]

    _now = now()
    filtered_cancels = []
    for days in range(conf.SUBURBAN_MOVISTA_CANCELS_FETCH_DAYS):
        request_dt = _now + timedelta(days=days)

        try:
            movista_cancels = movista_client.cancels(request_dt)
        except Exception:
            raise

        raw_cancels = map(cancel_to_db_raw, movista_cancels)
        log.info('{} movista cancels fetched on {}'.format(len(raw_cancels), request_dt))

        snapshot = get_snapshot_by_date(request_dt)
        cancels = filter_existing_cancels(raw_cancels, snapshot)
        log.info('{} ({} annulled) filtered movista cancels on {}'.format(
            len(cancels), len([c for c in cancels if c.from_express_id is None]), request_dt
        ))

        filtered_cancels.extend(cancels)

    return filtered_cancels


def filter_existing_cancels(cancels, snapshot):
    # type: (List[MovistaCancelRaw], Dict[unicode, MovistaCancelRaw]) -> List[MovistaCancelRaw]

    cancels_by_number = {}
    for cancel in cancels:
        cancels_by_number[cancel.train_number] = cancel

    cancelled_numbers = set(cancels_by_number.keys()) | set(snapshot.keys())

    filtered_cancels = []
    for number in cancelled_numbers:
        new_cancel = cancels_by_number.get(number)
        prev_cancel = snapshot.get(number)

        if new_cancel is None:
            if not (prev_cancel is None or prev_cancel.from_express_id is None):
                filtered_cancels.append(create_annulled_cancel(prev_cancel))
        elif prev_cancel is None or prev_cancel.from_express_id is None:
            filtered_cancels.append(new_cancel)
        elif new_cancel.create_dt > prev_cancel.create_dt:
            filtered_cancels.append(new_cancel)

    return filtered_cancels


def get_snapshot_by_date(dt):
    # type: (datetime) -> Dict[unicode, MovistaCancelRaw]

    cancels_by_number = {}
    cancels = list(MovistaCancelRaw.objects.filter(departure_date=dt.date()).order_by('create_dt'))

    for cancel in cancels:
        cancels_by_number[cancel.train_number] = cancel

    return cancels_by_number


def update_movista_raw_cancels():
    # type: () -> None

    with log_run_time('update movista raw cancels'):
        cancels = fetch_new_movista_cancels()
        if cancels is None:
            return

        log.info('{} new movista raw cancels'.format(len(cancels)))
        if cancels:
            MovistaCancelRaw.objects.insert(cancels)
