# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from collections import defaultdict
from functools import partial

import six

from common.apps.suburban_events import dynamic_params
from common.apps.suburban_events.forecast import match_rzd
from common.apps.suburban_events.forecast.events import prepare_events, save_prepared_events, prepare_mczk_events
from common.apps.suburban_events.forecast.forecast import recalc_forecasts
from common.apps.suburban_events.models import LVGD01_TR2PROC_feed
from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.logging import log_run_time


log = logging.getLogger(__name__)
log_run_time = partial(log_run_time, logger=log)


def raw_event_data_to_str(event_data):
    return u'{}\t'.format(event_data['_id']) + u'\t'.join(
        six.text_type(event_data[field])
        for field in LVGD01_TR2PROC_feed.fields_list()
    )


def log_matching_problems(not_matched_events):
    problems = []

    for e in not_matched_events:
        for problem in e.station_match_problems:
            problems.append((problem, e))

    for e in not_matched_events:
        if not e.station_match_problems:
            for problem in e.thread_match_problems:
                problems.append((problem, e))

    problems = sorted(problems, key=lambda pe: pe[0]['problem'])

    total = defaultdict(int)
    for problem, event in problems:
        total[problem['problem']] += 1
        log.info(u'{}'.format(problem['description']))
        log.info(raw_event_data_to_str(event.event_data))

    for k, v in total.items():
        log.info('total {}: {}'.format(k, v))


def update_all():
    """
    - проверяем наличие новых событий от любых источников (сейчас есть только РЖД)
    - матчим события на наши данные
    - сохраняем матченные события в прогнозатор
    - запускаем прогнозатор, который вычисляет и сохраняет состояния (прогнозы и факты)
    """

    with log_run_time('update suburban events', logger=log):
        last_event_id = dynamic_params.get_param('last_matched_rzd_event_id')

        with log_run_time('get new rzd events from {}'.format(last_event_id)):
            new_rzd_events, new_mczk_rzd_events = match_rzd.get_new_events(last_event_id)

        last_event_ids = []

        if new_rzd_events:
            with log_run_time('match {} new rzd events'.format(len(new_rzd_events))):
                matcher = match_rzd.EventsMatcher(new_rzd_events, log)
                matched_events, not_matched_events = matcher.match()
                log_matching_problems(not_matched_events)

            log.info(u'total {}, matched {}, not matched {} ({:.0f}%)'.format(
                len(new_rzd_events),
                len(matched_events),
                len(not_matched_events),
                100.0 * len(not_matched_events) / len(new_rzd_events),
            ))

            last_event_ids.append(new_rzd_events[-1]['_id'])
        else:
            matched_events = []

        if new_mczk_rzd_events and conf.SUBURBAN_ENABLE_MCZK:
            with log_run_time('match {} new mczk events'.format(len(new_mczk_rzd_events))):
                mczk_matcher = match_rzd.MCZKEventsMatcher(new_mczk_rzd_events, log)
                mczk_matched_events, mzck_not_matched_events = mczk_matcher.match()
                log_matching_problems(mzck_not_matched_events)

            log.info(u'total mczk {}, matched mczk {}, not mczk matched {} ({:.0f}%)'.format(
                len(new_mczk_rzd_events),
                len(mczk_matched_events),
                len(mzck_not_matched_events),
                100.0 * len(mzck_not_matched_events) / len(new_mczk_rzd_events),
            ))

            last_event_ids.append(new_mczk_rzd_events[-1]['_id'])
        else:
            mczk_matched_events = []

        with log_run_time('save {} matched suburban rzd events, save {} matched mczk rzd events'.format(
                len(matched_events), len(mczk_matched_events))):

            new_events, new_mczk_events = [], []
            for rzd_event in matched_events:
                new_events.extend(rzd_event.get_result_events())

            for mczk_event in mczk_matched_events:
                new_mczk_events.extend(mczk_event.get_result_events())

            log.info('Total {} suburban and {} mczk new events generated'.format(len(new_events), len(new_mczk_events)))
            prepared_events = prepare_events(new_events)
            prepared_events.update(prepare_mczk_events(new_mczk_events))
            save_prepared_events(prepared_events)

            if last_event_ids:
                last_event_id = max(last_event_ids)
                dynamic_params.set_param('last_matched_rzd_event_id', last_event_id)

        with log_run_time('recalc forecasts'):
            recalc_forecasts()
